package Jcg.viewer.processing3d;

import java.awt.Color;
import java.util.Collection;
import java.util.List;

import Jcg.geometry.Point_3;
import Jcg.geometry.Triangle_3;
import Jcg.mesh.*;
import Jcg.polyhedron.*;
import Jcg.viewer.DrawMesh;
import processing.core.PApplet;

/**
 * Class for drawing a surface mesh (Polyhedron_3) in 3D, using a Processing PApplet
 * 
 * @author Luca Castelli Aleardi (INF562/555, 2012-20)
 *
 */
public class DrawPolyhedronProcessing extends DrawMesh {

	/** Processing (P3D) frame for rendering 3D objects (meshes, faces, edges, points, ...) */
	PApplet view;
	
	/** the triangle mesh (to be rendered) */
	public Polyhedron_3<Point_3> polyhedron3D;
	
	/** the triangle soup (to be rendered) */
	public Collection<Triangle_3> triangleSoup; // 
	
	public float[][] edgeColors=null;
	public float[][] vertexNormals=null;
	
	/** a basic set of predefined colors for drawing faces, edges and vertices */
	public Color[] colors;
	
	/**
	 * Create a surface mesh
	 */	
	public DrawPolyhedronProcessing(PApplet view, Polyhedron_3<Point_3> polyhedron3D) {
		this.createColors();
		
		this.view=view;
    	this.polyhedron3D=polyhedron3D;
    	this.polyhedron3D.isValid(false);
	}
	
	/**
	 * Create a surface mesh from an OFF file
	 */	
	public DrawPolyhedronProcessing(PApplet view, String filename) {
		this.createColors();
		
		this.view=view;
    	polyhedron3D=MeshLoader.getSurfaceMesh(filename);
    	polyhedron3D.isValid(false);
	}

	/**
	 * Create a surface mesh from a 3D polyhedron (3D surface mesh). Edges are endowed with colors (RGB format)
	 */	
	public DrawPolyhedronProcessing(MeshViewerProcessing view, Polyhedron_3<Point_3> polyhedron3D, float[][] edgeColors) {
		this.createColors();
		
		this.view=view;
    	this.polyhedron3D=polyhedron3D;
    	this.polyhedron3D.isValid(false);
    	
    	this.edgeColors=edgeColors;
	}

	/**
	 * Create a 3D triangle soup
	 */	
	public DrawPolyhedronProcessing(MeshViewerProcessing view, Collection<Triangle_3> triangles) {
		this.createColors();
		
		this.view=view;
		
		this.triangleSoup=triangles;
	}
	

	/**
	 * Define a basic set of 13 colors for drawing edges, faces, vertices.
	 */	
	public void createColors() {
		this.colors=new Color[13];
		this.colors[0]=null; // no color defined
		this.colors[1]=Color.red;
		this.colors[2]=Color.blue;
		this.colors[3]=Color.black;
		this.colors[4]=Color.green;
		this.colors[5]=Color.orange;
		this.colors[6]=Color.cyan;
		this.colors[7]=Color.yellow;
		this.colors[8]=Color.magenta;
		this.colors[9]=new Color(240, 240, 240, 255); // standard color for rendering faces
		this.colors[10]=new Color(0, 128, 0, 255); // dark green
		this.colors[11]=new Color(180, 180, 180, 255); // dark gray
		this.colors[12]=new Color(0, 255, 0, 255); // light green gray		
	}

	/**
	 * Draw a segment between two points
	 */	
	public void drawSegment(Point_3 p, Point_3 q) {
		float s=1f;
		float x1=(float)p.getX().doubleValue()*s;
		float y1=(float)p.getY().doubleValue()*s;
		float z1=(float)p.getZ().doubleValue()*s;
		float x2=(float)q.getX().doubleValue()*s;
		float y2=(float)q.getY().doubleValue()*s;
		float z2=(float)q.getZ().doubleValue()*s;
		
		this.view.fill(255f, 0f, 255f, 255);
		this.view.line(	x1, y1, z1, x2, y2, z2 );		
	}

	/**
	 * Draw the vertex normal
	 */	
	public void drawNormal(Point_3 p, float[] normal) {
		float x1=(float)p.getX().doubleValue();
		//System.out.println("vertex normal "+p);
		float y1=(float)p.getY().doubleValue();
		float z1=(float)p.getZ().doubleValue();
		float x2=(x1+normal[0]);
		float y2=(y1+normal[1]);
		float z2=(z1+normal[2]);
		
		this.drawSegment(p, new Point_3(x2, y2, z2), new float[]{0f, 255f, 0f, 255f});
	}

	/**
	 * Draw a segment between two points
	 */	
	public void drawSegment(Point_3 p, Point_3 q, float[] color) {
		float s=1f;
		float x1=(float)p.getX().doubleValue()*s;
		float y1=(float)p.getY().doubleValue()*s;
		float z1=(float)p.getZ().doubleValue()*s;
		float x2=(float)q.getX().doubleValue()*s;
		float y2=(float)q.getY().doubleValue()*s;
		float z2=(float)q.getZ().doubleValue()*s;
		
		this.view.stroke(color[0], color[1], color[2], color[3]);
		this.view.line(	x1, y1, z1, x2, y2, z2 );		
	}

	/**
	 * Draw a vertex (as a small sphere)
	 */	
	public void drawVertex(Point_3 p) {
		float s=1f;
		float x1=(float)p.getX().doubleValue()*s;
		float y1=(float)p.getY().doubleValue()*s;
		float z1=(float)p.getZ().doubleValue()*s;
		
		view.translate(x1, y1, z1);
		view.sphere(s/25f);
		view.translate(-x1, -y1, -z1);
	}

	/**
	 * Draw a triangle
	 */	
	public void drawTriangle(Point_3 p, Point_3 q, Point_3 r) {
		float s=1f;
		view.vertex( (float)(p.getX().doubleValue()*s), (float)(p.getY().doubleValue()*s), (float)(p.getZ().doubleValue()*s));
		view.vertex( (float)(q.getX().doubleValue()*s), (float)(q.getY().doubleValue()*s), (float)(q.getZ().doubleValue()*s));
		view.vertex( (float)(r.getX().doubleValue()*s), (float)(r.getY().doubleValue()*s), (float)(r.getZ().doubleValue()*s));
	}
	
	/**
	 * Draw a (triangle or polygonal) face
	 */	
	public void drawFace(Point_3[] f) {
		System.out.println("Warning: this method is not implemented");
	}

	/**
	 * Draw a (triangle or polygonal) face
	 */	
	public void drawFace(Face<Point_3> f) {
		if(f==null )
			return;
		
		if(this.colors[f.tag]==null)
			this.drawFace(f, this.colors[9]); // color for standard faces 
		else
			this.drawFace(f, this.colors[f.tag]); // draw the faces with a given color
	}
	
	/**
	 * Draw a (triangle or polygonal) face of a Polyhedral surface
	 */	
	public void drawFace(Face<Point_3> f, Color c) {
		Halfedge<Point_3> h=f.getEdge();
		Halfedge<Point_3> pEdge=h.getNext();
		
		Point_3 u=h.getOpposite().getVertex().getPoint();
		view.noStroke();
		view.fill((float)c.getRed(), (float)c.getGreen(), (float)c.getBlue()); // color of the triangle
		
		while(pEdge.getVertex()!=h.getOpposite().getVertex()) {
			Point_3 v=pEdge.getOpposite().getVertex().getPoint();
			Point_3 w=pEdge.getVertex().getPoint();
			this.drawTriangle(u, v, w); // draw a triangle face
			
			pEdge=pEdge.getNext();
		}
	}
	
	/**
	 * Draw the entire mesh
	 */
	public void drawMesh(int type) {
		//this.drawAxis();
		
		// draw all faces
		view.beginShape(view.TRIANGLES);
		this.view.fill(100f, 100f, 100f, 50f);
		for(Face<Point_3> f: this.polyhedron3D.facets) {
				this.drawFace(f);
		}
		view.endShape();
		
		if(type==1) return; // no rendering of edges
		
		// draw all edges
		view.strokeWeight(4); // line width (for edges)
		view.stroke(20);
		
		for(Halfedge<Point_3> e: this.polyhedron3D.halfedges) {
			Point_3 p=e.vertex.getPoint();
			Point_3 q=e.opposite.vertex.getPoint();
			
			if(this.edgeColors==null) // edge colors are not defined
				this.drawSegment(p, q); // draw edge (p,q)
			else /*if(this.edgeColors[e.index][0]==255f)*/ {	
				this.drawSegment(p, q, this.edgeColors[e.index]);
			}
		}
		//view.strokeWeight(1);
		
		// draw vertex normals, if they are defined
		if(this.vertexNormals!=null) {
			view.stroke(0f, 250f, 0f);
			int i=0;
			for(Vertex<Point_3> v: this.polyhedron3D.vertices) {
				Point_3 p=v.getPoint();
				this.drawNormal(p, this.vertexNormals[i]);
				i++;
			}
		}
		
		if(type==2) return; // no rendering for vertices
		
		view.noStroke();
		view.fill(0f, 0f, 250f);
		for(Vertex<Point_3> v: this.polyhedron3D.vertices) {
			Point_3 p=v.getPoint();
			this.drawVertex(p);
		}

		view.strokeWeight(1);
	}
	
	/**
	 * Draw the triangle soup
	 */
	public void drawTriangleSoup(int type) {
		//System.out.println("Drawing trangle soup");
		//this.drawAxis();
		
		// draw all faces
		view.fill(200f, 200f, 200f);
		view.beginShape(view.TRIANGLES);
		for(Triangle_3 t: this.triangleSoup) {
			this.drawTriangle(t.vertex(0), t.vertex(1), t.vertex(2));
		}
		view.endShape();
		
		view.noStroke();
		view.fill(0f, 0f, 250f);
		for(Triangle_3 t: this.triangleSoup) {
			this.drawVertex(t.vertex(0));
			this.drawVertex(t.vertex(1));
			this.drawVertex(t.vertex(2));
		}
		view.strokeWeight(1);

	}
	
	/**
	 * Draw the 3D scene
	 */
	public void draw(int type) {
		if(this.polyhedron3D!=null)
			this.drawMesh(type);
		if(this.triangleSoup!=null)
			this.drawTriangleSoup(type);
	}
	
	/**
	 * Compute the scale factor (depending on the max distance of the point set)
	 */
	public float computeScaleFactor() {
		double maxDistance=1.;
		Point_3 origin=new Point_3(0., 0., 0.);
		if(this.polyhedron3D!=null && this.polyhedron3D.vertices.size()>=1) {
			for(Vertex<Point_3> v: this.polyhedron3D.vertices) {
				double distance=Math.sqrt(v.getPoint().squareDistance(origin).doubleValue());
				maxDistance=Math.max(maxDistance, distance); 
			}
		}
		return (float)(Math.sqrt(3)/maxDistance*150);
	}

	/**
	 * Reset face/edge/vertex colors. <br>
	 * Set the initial tag of all faces, edges and vertices to a given value (used for color rendering)
	 */	
	public static void resetMeshColors(Polyhedron_3 mesh) {
		for(Vertex v: (List<Vertex>)mesh.vertices)
			v.tag=0;
		for(Face f: (List<Face>)mesh.facets) {
			f.tag=0; // standard face color
		}
		for(Halfedge e: (List<Halfedge>)mesh.halfedges) {
			e.tag=0;
		}
	}

}
