package Jcg.util;
import java.util.ArrayList;

/** This class provides basic functions for computing some statistical parameters */
public class Statistics {

	/**
	 * Normalized the input data with respect to a given value <br>
	 */
	public static double[] normalize(double[] x, double norm){
		if(x==null)
			return null;
		
		int n=x.length;
		double[] result=new double[n];
		for(int i=0; i<n; i++){
			result[i]=x[i]/norm;
		}
		return result;
	}

	/**
	 * Compute the average of the (non negative) values in the array <br>
	 * Remark: negative values are skipped
	 */
	public static double getAverage(double[] x){
		if(x==null)
			return -1.;
		
		double avg=0.0;
		int count=0;
		for(int i=0; i<x.length; i++){
			if(x[i]>=0.0) { // count only positive terms
				count++;
			}
		}
		for(int i=0; i<x.length; i++){
			if(x[i]>=0.0) { // count only positive terms
				avg=avg+(x[i]/count);
			}
		}
		return avg;
	}
	
	/**
	 * Compute the maximum of the value in the array
	 */
	public static double getMax(double[] x){
		if(x==null)
			return -1.;
		
		double max=Double.MIN_VALUE;
		for(int i=0;i<x.length;i++){
			if(x[i]>max) {
				max=x[i];
			}
		}
		return max;
	}
	
	/**
	 * Compute the minimum of the value in the array
	 */
	public static double getMin(double[] x){
		if(x==null) {
			System.out.println("\t --- Warning: array not defined ---");
			return Double.MAX_VALUE;
		}
		
		double min=Double.MAX_VALUE;
		for(int i=0;i<x.length;i++){
			if(x[i]<min) {
				min=x[i];
			}
		}
		return min;
	}

	/**
	 * Compute the average percent deviation
	 * <br>
	 * Remark: negative values are skipped
	 */
	public static double getDeviation(double[] x) {
		if(x==null)
			return -1.;
		
		int n=x.length;
		double d=0.0;
		double avg=getAverage(x);
		double max=getMax(x);
		
		int nonNegative=0;
		for(int i=0;i<n;i++){
			if(x[i]>=0.0) {
				double term=Math.abs((x[i]-avg)/(Math.max(avg, max-avg)));
				d=d+term;
				nonNegative++;
			}
		}
		
		d=d/nonNegative;
		
		//System.out.println("max length="+max+", avg length="+avg+", edge length aesthetic="+(1-d));
		return (1-d);
	}

	/**
	 * Compute the average percent deviation
	 * <br>
	 * Remark: only (strictly) positive values are considered
	 */
	public static double getDeviationOnlyPositive(double[] x) {
		if(x==null)
			return -1.;
		
		int n=x.length;
		double d=0.0;
		double avg=getAverage(x);
		double max=getMax(x);
		
		int positive=0;
		double den=Math.max(avg, max-avg);
		for(int i=0;i<n;i++){
			if(x[i]>0.0) {
				double term=Math.abs((x[i]-avg)/den);
				//if(i<100)
				//	System.out.println(x[i]+": "+term);
				d=d+term;
				positive++;
			}
		}
		
		d=d/positive;
		
		//System.out.println("max length="+max+", avg length="+avg+", edge length aesthetic="+(1-d));
		return (1-d);
	}

	/**
	 * Given a sorted array as input, it returns the following 5 values: <br>
	 * -) 0 percentile (minimum) <br>
	 * -) 25 percentile <br>
	 * -) 50 percentile (median) <br>
	 * -) 75 percentile <br>
	 * -) 100 percentile (maximum) <br>
	 * 
	 * @return  an array of size 5
	 */    
	public static double[] getPercentile(double[] data) {
		if(data==null || data.length==0)
			return null;		

		int n=data.length;
		double min=data[0];
		double perc25=data[n/4];		
		double median=data[n/2];		
		double perc75=data[n/4*3];		
		double max=data[n-1];
		
		return new double[] {min, perc25, median, perc75, max};
    }

	/**
	 * Compute the Shannon entropy (0-entropy) of an integer input sequence. Values are assumed to range in [0..n]
	 * 
	 * @param s  	input sequence
	 * @param n		maximum value in the sequence: all values are in [0..n]
	 */
	public static double getEntropyInteger(int[] s, int n) {
		if(s==null) {
			System.out.println("Error: input sequence not defined");
			return -1;
		}
		
		double entropy = 0.0; // the shannon entropy
        int[] frequencies = new int[n+1];
        
        // compute symbol frequencies
        for (int j = 0; j < n; j++) {
        	frequencies[s[j]]++;
        }

        // compute Shannon entropy
        for (int i = 0; i < n; i++) {
        	double p=0.0;
            if (frequencies[i] > 0) {
            	p = ((double)frequencies[i]) / n;
            	entropy -= p * Math.log(p) / Math.log(2);
            }
        }

        
		return entropy;
	}
	
	/**
	 * Return the Latex code representing a bar chart
	 * 
	 * @param t an ordered array containing 5 values [min, perc25, median, perc75, max]
	 * @return a string representing the bar chart (according to latex Style)
	 */
	public static String minMaxToLatexChart(double[] t) {
		double min=approx(t[0]);
		double max=approx(t[4]);
		
		String result=approx((max+min)/2.)+" 0 "+approx((max-min)/2.);
		return result;
	}
	
	/**
	 * Return the Latex code representing a bar chart
	 * 
	 * @param t an ordered array containing 5 values [min, perc25, median, perc75, max]
	 * @return a string representing the bar chart (according to latex Style)
	 */
	public static String percentile2575ToLatexChart(double[] t) {
		double p25=approx(t[1]);
		double p75=approx(t[3]);
		
		String result=approx((p75+p25)/2.)+" 0 "+approx((p75-p25)/2.);
		return result;
	}
	
	/**
	 * Return the Latex code for drawing the whisker plot of the input data. There are tow kind of boxes: <br>
	 * -) thin box: representing the minimum and maximum values (for the y-range) <br>
	 * -) thick box: representing the percentiles in the range [25%,75%] of the values stored in the array 'y[i]' <br>
	 * 
	 * @param x  an array corresponding to the x-coordinates of the 'n' vertical bars
	 * @param y  a double array containing the values to plot. The i-th array 'y[i]' is assumed to be sorted
	 */    
	public static String latexWhiskerPlot(double[] x, double[][] y) {
		String header1pt="\\addplot [scatter, color=red, only marks, mark=none]" + 
				" plot [error bars/.cd, y dir = both, y explicit, error bar style={line width=1pt}]\n" + 
				" table[row sep=crcr, x index=0, y index=1, x error index=2, y error index=3,]{\n";
		String header2pt="\\addplot [scatter, color=red, only marks, mark=none]" + 
				" plot [error bars/.cd, y dir = both, y explicit, error bar style={line width=2.5pt}]\n" + 
				" table[row sep=crcr, x index=0, y index=1, x error index=2, y error index=3,]{\n";
		
		String result1=header1pt; // first (thin) bars
		String result2=header2pt; // first (thick) bars
		double[] values, perc;
		int n=x.length;
		for(int i=0;i<n;i++) {
			if(y[i]==null || y[i].length==0)
				continue;
			
			perc=Statistics.getPercentile(y[i]);
			if(perc!=null) {
				result1=result1+approx(x[i]); // midpoint in the corresponding range
				result2=result2+approx(x[i]); // as above
				//System.out.println("Percentiles: "+perc[0]+" - "+perc[1]+" - "+perc[2]+" - "+perc[3]+" - "+perc[4]+" - ");
				result1=result1+" "+Statistics.minMaxToLatexChart(perc)+"\\\\ ";
				result2=result2+" "+Statistics.percentile2575ToLatexChart(perc)+"\\\\ ";
			}
		}
		result1=result1+"\n};\n";
		result2=result2+"\n};\n";
		
		System.out.println("Latex code written");
		return result1+"\n"+result2;
	}

	private static double approx(double x) {
		double prec=3; // numerical precision
		
		double p=(int)Math.pow(10, prec);
		return ((int)(x*p))/p;
	}

	private static double approx(double x, int prec) {
		double p=(int)Math.pow(10, prec);
		return ((int)(x*p))/p;
	}

}
