package Jcg.util;

import java.util.LinkedList;
import java.util.List;

import Jcg.polyhedron.Halfedge;
import Jcg.polyhedron.Polyhedron_3;
import Jcg.polyhedron.Vertex;

/**
 * This class provides methods for computing mesh statistics (average degree, diameter, ...). 
 * 
 * @author Luca Castelli Aleardi (Ecole Polytechnique, 2019-2025)
 */
public class MeshStatistics {
	/** Input mesh (half-edge representation) */
	Polyhedron_3 mesh;
	
	private int precision=2;
	public int verbosity=1;
	
	public int[] eccentricity=null;
	
	public MeshStatistics(Polyhedron_3 m) {
		this.mesh=m;
	}
	
	/**
	 * Return the number of nodes of the graph
	 */
	public int sizeVertices() {
		return this.mesh.sizeOfVertices();
	}
	
	/**
	 * Compute the proportion of vertices having a given degree.
	 * 
	 * @param  degree of a vertex
	 * @return  the proportion of vertices having a given 'degree' 
	 */
	public double statsVertexDegree(int degree) {
		List<Vertex> vertices=mesh.vertices;
		double n=vertices.size();
		
		double counter=0;
		for(Vertex v: vertices) {
			if(mesh.vertexDegree(v)==degree)
				counter++;
		}
		
		return counter/n;
    }

	/** 
	 * Return the vertex degree distribution, up to degree 'max' 
	 **/
	public double[] getVertexDegreeDistribution() {
		List<Vertex> vertices=mesh.vertices;
		double n=vertices.size();
		
		double[] result=new double[mesh.vertices.size()];
		for(Vertex v: vertices) { // It can be improved here (iterate over half-edges)
			int d=mesh.vertexDegree(v);
			if(d<n)
				result[d]++;
		}
		for(int i=0;i<n;i++)
			result[i]=result[i]/n;
		
		return result;
    }

	/** 
	 * Return a String storing some mesh statistics: <br>
	 * -) minimum and maximum vertex degree <br>
	 * -) the vertex degree distribution, up to degree 'max' <br>
	 * -) number of separating triangles, if needed <br>
	 * 
	 * @param verbosity  	if verbosity>0 show the heading line
	 **/
	public String meshStatisticsToString(int maxDegree, boolean sepTriangles, int verbosity) {
		int precision=4; // numeric precision
		int N=mesh.sizeOfVertices();
		int E=mesh.sizeOfHalfedges()/2;
		int genus=this.mesh.genus();
		int minDeg=N-1, maxDeg=0;
		double[] degrees=new double[N];
		String result="";
		
		double[] degreeSequence=this.getVertexDegreeSequence();
		double avgPercentDeviation=Statistics.getDeviationOnlyPositive(degreeSequence);
		
		List<Vertex> vertices=mesh.vertices;
		for(Vertex v: vertices) { // It can be improved here (iterate over half-edges)
			int d=mesh.vertexDegree(v);
			degrees[d]++;
			minDeg=Math.min(minDeg, d);
			maxDeg=Math.max(maxDeg, d);
		}
		for(int i=0;i<N;i++)
			degrees[i]=degrees[i]/N;
		
		double avgVG=this.getAverageVertexGap();
		double avgBW=this.getAverageBandwidth();
		double entropyDegreeSequence=this.getVertexDegreeEntropy();
		
		// initialize the final result
		result=result+N+"\t"+E+"\t"+genus+"\t"+this.type();
		result=result+"\t"+this.approx(avgVG, 1)+"\t"+this.approx(avgBW, 1);
		result=result+"\t"+minDeg+"\t"+maxDeg+"\t"+this.approx(avgPercentDeviation, precision);
		result=result+"\t"+this.approx(entropyDegreeSequence, precision);
		for(int i=2;i<=Math.min(maxDegree, maxDeg);i++)
			result=result+"\t"+approx(degrees[i], precision);
		if(sepTriangles==true) {
			int separating=mesh.getSeparatingTriangles().size();
			double proportion=(100.*separating)/mesh.sizeOfFacets();
			result=result+"\t"+separating+"\t"+this.approx(proportion, precision);
		}
		
		if(verbosity>0) // add the heading line
			return headingToString(maxDegree, sepTriangles)+"\n"+result;
		else
			return result;
    }

	/** 
	 * Return a the heading: <br>
	 * n	e	genus	type	minDeg	maxDeg	avgDev	deg2	deg3	...		sepTri	sepTri% <br>
	 **/
	public static String headingToString(int max, boolean separatingTriangles) {
		String heading="";

		// initialize the heading string
		heading=heading+"n"+"\te"+"\tgenus"+"\ttype";
		heading=heading+"\tavgVG"+"\tavgBW";
		heading=heading+"\tminDeg"+"\tmaxDeg"+"\tavgDev"+"\tentDeg";
		for(int i=2;i<=max;i++)
			heading=heading+"\tdeg"+i;
		if(separatingTriangles==true)
			heading=heading+"\tsepTri"+"\tsepTri%";
			
		return heading;
    }

	/**
	 * Return the vertex degree sequence (int[] array) <br>
	 * <br>
	 * <b>Warning</b>: the runtime complexity is not optimal (iterating over edges could be slightly faster)
	 * 
	 * @return an array of size N, storing the vertex degree sequence
	 */
	public static int[] getVertexDegrees(Polyhedron_3 tri) {
		List<Vertex> vertices=tri.vertices;
		int n=vertices.size();
		
		int[] result=new int[n];
		for(Vertex v: vertices) {
			int d=tri.vertexDegree(v);
			result[v.index]=d;
		}
		
		return result;
    }

	/**
	 * Return the vertex degree sequence (double[] array)
	 * @return an array of doubles of size N, storing the vertex degree sequence
	 */
	public double[] getVertexDegreeSequence() {
		List<Vertex> vertices=mesh.vertices;
		int n=vertices.size();
		
		double[] result=new double[n];
		for(Vertex v: vertices) {
			int d=mesh.vertexDegree(v);
			result[v.index]=d;
		}
		
		return result;
    }

	/**
	 * Return the entropy of the vertex degree sequence <br>
	 * 
	 * @return the entropy of the vertex degree sequence
	 */
	public double getVertexDegreeEntropy() {
		int[] degrees=getVertexDegrees(this.mesh); // vertex degree sequence
		int max=0;
		for(int i=0;i<degrees.length;i++) {
			if(degrees[i]>0)
				max=i;
		}
		double result=Statistics.getEntropyInteger(degrees, max);
		//System.out.println("entropy degree sequence= "+result);
		return result;
    }

	/**
	 * Return the average degree of the graph
	 */
	public double averageDegree() {
		System.out.println("To be completed");
		return 0;
	}

	/**
	 * Return the number of connected components of the graph
	 */
	public int numberConnectedComponents() {
		System.out.println("To be completed");
		return 0;
	}

	/**
	 * Get the type of the mesh: triangle, quad, polygonal
	 */
	public String type() {
		if(this.mesh.isPureTriangle())
			return "tri";
		else if(this.mesh.isPureQuad())
			return "quad";
		else return "poly";
	}

    /**
     * Compute, from each vertex, the (graph) distance to vertex v
     * It performs a BFS visit of the entire mesh starting from vertex v
     * <p>
     * Remark: the vertices are assumed to have an index, between 0..n-1
     * 
     * @param v  the starting vertex
     * @return an array containing the (integer) distances from vertex v
     */		   
    public int computeAllDistancesFromVertex(Vertex v){
    	if(v==null)
    		return -1;
    	
    	int n=mesh.sizeOfVertices(); // number of vertices in the mesh
    	
    	int[] distance=new int[n]; // it stores, for each vertex the distance from the source
    	LinkedList<Vertex> queue=new LinkedList<Vertex>(); // stack containing the node to visit
    	
    	for(int i=0;i<n;i++)
    		distance[i]=-1; // unvisited vertices have distance -1 from the source vertex 'v'
    	
    	distance[v.index]=0;
    	queue.add(v);
    	while(queue.isEmpty()==false) {    			
    		Vertex u=queue.poll(); // get and removes the node in the head of the stack
    		//System.out.println("\n current node: "+u.index);

    		List<Halfedge> neighbors=u.getOutgoingHalfedges();
    		for(Halfedge e: neighbors) { // visit incident (outgoing) halfedges around 'u'
    			Vertex neighbor=e.getVertex();
    			if(distance[neighbor.index]==-1) {
    				distance[neighbor.index]=distance[u.index]+1; // increment the distance by 1
    				queue.add(neighbor);
    			}
    		}
    	}
    	
    	int max=0;
    	for(int i=0;i<n;i++)
    		max=Math.max(max, distance[i]);
    	
    	return max;
    }

    /**
     * Compute an approximation of the diameter using a small set of random seed vertices
     * <p>
     * Remark: the vertices are assumed to have an index, between 0..n-1
     * 
     * @param mesh  input polyhedral mesh
     * @return the approximated diameter
     */		   
    public int approximatedDiameter(int seeds){
    	int n=mesh.sizeOfVertices(); // number of vertices in the mesh
    	int diameter=0;
    	
    	System.out.print("Computing approximated diameter ("+seeds+" random seeds)...");
		long startTime=System.nanoTime(), endTime; // for evaluating time performances

    	for(int i=0;i<seeds;i++) {
    		int index=(int)(Math.random()*n);
    		
    		Vertex u=(Vertex)mesh.vertices.get(index);
    		int distance=computeAllDistancesFromVertex(u);
    		diameter=Math.max(diameter, distance);
    	}
    	
    	endTime=System.nanoTime();
        double duration=(double)(endTime-startTime)/1000000000.;
		
        System.out.println("done ("+duration+" seconds)");

    	return diameter;
    }

    /**
     * Compute the exact diameter with brute force (slow)
     * <p>
     * Remark: the vertices are assumed to have an index, between 0..n-1
     * 
     * @return the exact diameter
     */		   
    public int diameter(){
    	int n=mesh.sizeOfVertices(); // number of vertices in the mesh
    	int diameter=0;
    	
    	System.out.print("Computing exact diameter (brute force)...");
		long startTime=System.nanoTime(), endTime; // for evaluating time performances

		List<Vertex> vertices=mesh.vertices;
    	for(Vertex u: vertices) {
    		int distance=computeAllDistancesFromVertex(u);
    		diameter=Math.max(diameter, distance);
    	}
    	
    	endTime=System.nanoTime();
        double duration=(double)(endTime-startTime)/1000000000.;
		
        System.out.println("done ("+duration+" seconds)");

    	return diameter;
    }

    /**
     * Compute the exact diameter with brute force <br>
     * <br>
     * Warning: this method can be slow for large meshes (>10k vertices) <br>
     * <p>
     * Remark: the vertices are assumed to have an index, between 0..n-1
     * 
     * @return the approximated diameter
     */		   
    public void bruteForceEccentricity(){
    	int n=mesh.sizeOfVertices(); // number of vertices in the mesh
    	this.eccentricity=new int[n];
    	
    	if(this.verbosity>0)
    		System.out.print("Computing vertex eccentrivities (brute force)...");
		long startTime=System.nanoTime(), endTime; // for evaluating time performances

		List<Vertex> vertices=mesh.vertices;
    	for(Vertex u: vertices) {
    		eccentricity[u.index]=computeAllDistancesFromVertex(u);
    	}
    	
    	endTime=System.nanoTime();
        double duration=(double)(endTime-startTime)/1000000000.;
		
        if(this.verbosity>0)
        	System.out.println("done ("+duration+" seconds)");
    }

    /**
     * Return the 'radius' and 'diameter' of the graph
     */		   
    public int[] getRadiusAndDiameter(){
    	if(this.eccentricity==null) {
    		System.out.println("Warning: vertex eccentricity is not defined for all vertices");
    		return null;
    	}
    	
    	int n=this.eccentricity.length; // number of vertices in the mesh
    	
    	int diameter=0;
    	int radius=Integer.MAX_VALUE;

    	for(int i=0;i<n;i++) {
    		diameter=Math.max(diameter, this.eccentricity[i]);
    		radius=Math.min(radius, this.eccentricity[i]);
    	}

    	return new int[] {radius, diameter};
    }
    
    /**
     * Return the average <em>vertex gap</em> (as defined in Barik et al.). <br>
     * The gap of an edge (u, v) is the absolute value |u-v|.
     * <br> <br>
     * 
     * <b>Reference</b>: Vertex Reordering for Real-World Graphs and Applications: An Empirical Evaluation (Barik et al. 2020)
     * 
     * @return  the average vertex gap (over all edges)
     */
    public double getAverageVertexGap() {
    	double avgGap=0.0;
    	double E=this.mesh.sizeOfHalfedges()/2.;
    	
    	List<Halfedge> edges=this.mesh.halfedges;
    	for(Halfedge e: edges) {
    		double diff=e.vertex.index-e.getOpposite().vertex.index;
    		if(diff>0) // edges are counted only once
    			avgGap=avgGap+diff/E;
    	}
    	
    	return avgGap;
    }
    
    /**
     * Return the <em>average vertex gap (profile)</em> of a mesh whose vertices have been permuted according
     * to a given permutation 'pi' of [0...n-1]. <br>
     * <b>Remark</b>: vertex 'u' has number pi(u) in the permuted graph. <br>
     * The gap of an edge (u, v) is the absolute value |pi(u)-pi(v)|.
     * <br> <br>
     * 
     * <b>Reference</b>: Vertex Reordering for Real-World Graphs and Applications: An Empirical Evaluation (Barik et al. 2020)
     * 
     * @param pi	the input vertex permutation  	
     * @return  	the average vertex gap (over all edges) for a permuted mesh
     */
    public double getAverageVertexGap(int[] pi) {
    	double avgGap=0.0;
    	double E=this.mesh.sizeOfHalfedges()/2.;
    	int n=this.mesh.sizeOfVertices();
    	
    	if(pi==null || pi.length!=n)
    		throw new Error("Wrong vertex permutation: size "+pi.length);
    	
    	List<Halfedge> edges=this.mesh.halfedges;
    	for(Halfedge e: edges) {
    		if(pi[e.vertex.index]<0 || pi[e.vertex.index]>n-1)
    			throw new Error("Error: wrong permutation ");
    		if(pi[e.getOpposite().vertex.index]<0 || pi[e.getOpposite().vertex.index]>n-1)
    			throw new Error("Error: wrong permutation ");
    		double diff=pi[e.vertex.index]-pi[e.getOpposite().vertex.index];
    		if(diff>0) {// edges are counted only once
    			//System.out.print("e"+e.index+" ("+e.getVertex().index+","+e.getOpposite().getVertex().index+")\t gap="+diff); for debug
    			//System.out.println("\t"+pi[e.vertex.index]+","+pi[e.getOpposite().vertex.index]);
    			avgGap=avgGap+diff/E;
    		}
    	}
    	
    	return avgGap;
    }
    
    /**
     * Return the <em>vertex bandwidth</em> of a vertex v. <br>
     * The bandwidth is the maximal gap between vertex 'v' and its neighbors.
     * 
     * @param v  a vertex in a polygonal mesh
     */
    public int vertexBandwidth(Vertex v) {
    	int bandwidth=0;
    	Halfedge e=v.getHalfedge(); // halfedge incident to v (having v as destination)
    	
    	bandwidth=Math.abs(v.index-e.getOpposite().getVertex().index); // compute the gap for the first neighbor
    	
    	Halfedge pEdge=e.getNext().getOpposite();
    	while(pEdge!=e) {
    		bandwidth=Math.max(Math.abs(v.index-pEdge.getOpposite().getVertex().index), bandwidth); // get the opposite half-edge (outgoing from v)
    		pEdge=pEdge.getNext().getOpposite(); // next edge in cw order
    	}
    	return bandwidth;
    }
    
    /**
     * Return the average <em>vertex bandwidth</em>. <br>
     * 
     * @param v  a vertex in a polygonal mesh
     */
    public double getAverageBandwidth() {
    	double result=0.0;
    	double N=this.mesh.sizeOfVertices();
    	
    	List<Vertex> vertices=this.mesh.vertices;
    	for(Vertex v: vertices) {
    		double b=this.vertexBandwidth(v)/N;
    		result=result+b;
    		//System.out.println("v"+v.index+": "+this.vertexBandwidth(v)+", "+b); // only for debug
    	}
    	
    	return result;
    }
	
	public static double approx(double x, int prec) {
		double p=(int)Math.pow(10, prec);
		return ((int)(x*p))/p;
	}

}
