package Jcg.util;

/**
 * Minimal implementation of a doubly connected linked list, storing generic type elements. <br>
 * It uses sentinel (dummy and empty) nodes to represent the beginning and the end of the list.
 * 
 * @author Luca Castelli Aleardi (Ecole Polytechnique, 2013) <br>
 */
public class DLinkedList<X> {
	private DListNode<X> first, last;
	private int size;
	
	public DLinkedList() {
		this.first = new DListNode<X>(null, null, null);
		this.last = new DListNode<X>(null, null, first);
		this.first.next = last;
		size = 0;
	}

	/**
	 * Return a new linkedList, copy of the current list: only the content is copied (nodes are different)
	 */
	public DLinkedList<X> copy() {
		if(this.isEmpty())
			return new DLinkedList<X>();
			
		DLinkedList<X> result=new DLinkedList<X>();
		DListNode<X> n=this.getFirst();
		while(n!=this.last) {
			result.addLast(n.getElement());
			n=n.getNext();
		}
		
		// check for correctness
		if(this.size!=result.size)
			throw new Error("Warning: wrong copy of the list");
		if(this.getFirst().getElement()!=result.getFirst().getElement())
			throw new Error("Warning: wrong copy... first elements do NOT match");
		if(this.getLast().getElement()!=result.getLast().getElement())
			throw new Error("Warning: wrong copy... last elements do NOT match");
		
		return result;
	}

	/**
	 * Check equality between two lists: only the contents do count
	 */
	public boolean equal(DLinkedList<X> list) {
		if(this.size!=list.size)
			return false;
			
		DListNode<X> node1=this.getFirst();
		DListNode<X> node2=list.getFirst();
		while(node1!=this.last) {
			if(node1.getElement()!=node2.getElement()) // check the content of the tw nodes (in the two lists)
				return false;
			
			node1=node1.getNext();
			node2=node2.getNext();
		}
		
		return true;
	}
	
	public int size() { 
		return this.size; 
	}
	
	/**
	 * Check whether the list is empty
	 */
	public boolean isEmpty() {
		return this.size == 0; 
	}

	/**
	 * Return the 'start' of the list: useful for creating circular lists
	 */
	public DListNode<X> getStart() {
		return this.first;
	}

	/**
	 * Return the end of the list: useful for iterating over the nodes
	 */
	public DListNode<X> getEnd() {
		return this.last;
	}

	/**
	 * Return the first node of the list
	 * The result is a null reference if the list is empty
	 */
	public DListNode<X> getFirst() {
		return this.first.getNext();
	}

	/**
	 * Return the last node of the list
	 * The result is a null reference if the list is empty
	 */
	public DListNode<X> getLast() {
		return this.last.getPrev();
	}

	/**
	 * Insert a new element at the beginning of the list
	 * @param el the element to insert
	 */
	public void addFirst(X el) {
		insertAfter(first, el);
	}
	
	/**
	 * Append a new element at the end of the list
	 * @param el the element to insert
	 */
	public void addLast(X el) {
		insertAfter(last.previous, el);
	}
	
	/**
	 * Append a new element at the beginning of the list
	 * @param el the element to insert
	 */
	public void add(X el) {
		insertAfter(last.previous, el);
	}

	
	/**
	 * Append the cells of 'l2' to the current list. The merge is performed only when both lists are not empty.<br>
	 * Remark: this operation takes only constant time. <br>
	 * 
	 * Warnings: 
	 * -) both lists are modified
	 * 
	 * @param l2  the list to be added (at the end of the current list)
	 */
	public void append(DLinkedList<X> l2) {
		if(l2==null || l2.isEmpty())
			return;
		
		if(this.isEmpty()) { // do not perform the merge, when the current list is empty
			System.out.println("Warning: the merge should be not be performed when the current list is empty. Check it before.");
			return;
		}
		
		this.getLast().next=l2.getFirst();
		l2.getFirst().previous=this.last;
		this.size=this.size+l2.size;
		this.last=l2.last;
	}

	/**
	 * Insert a new element after a given node in the list
	 * @param el the element to insert
	 * @param nPnode the place where the element will be inserted
	 */
	public void insertAfter(DListNode<X> pNode, X el) {
		DListNode<X> item = new DListNode<X>(el, pNode.next, pNode);
		item.previous.next = item;
		item.next.previous = item;
		size++;
	}

	/**
	 * Insert a new element before a given node in the list
	 * @param el the element to insert
	 * @param nPnode the place where the element will be inserted
	 */
	public void insertBefore(DListNode<X> pNode, X el) {
		DListNode<X> item = new DListNode<X>(el, pNode, pNode.previous);
		item.previous.next = item;
		item.next.previous = item;
		size++;
	}
	
	/**
	 * Remove a given node from the list
	 * @param nPnode the node to delete
	 */
	public void delete(DListNode<X> pNode) {
		if(pNode==null)
			return;
		pNode.next.previous = pNode.previous;
		pNode.previous.next = pNode.next;
		size--;
	}

	/**
	 * Return a String representing the list
	 */
	public String toString() {
		if(this.isEmpty())
			return "empty list";
		String result="[";
		DListNode<X> n=this.getFirst();
		while(n!=this.last) {
			result=result+n.getElement()+", ";
			n=n.getNext();
		}
		return result+"]";
	}

}
