package Jcg.util;

/**
 * Minimal implementation of a circular doubly linked list, storing generic type elements
 * 
 * @author Luca Castelli Aleardi (Ecole Polytechnique, 2021-2024)
 */
public class CircularDLinkedList<X> {
	/** a reference to a node in the list (the HEAD) */
	public DListNode<X> first;
	
	/** number of nodes in the list */
	private int size;
	
	/** 
	 * Create a circular linked list from a doubly linked list (not circular). <br>
	 * <br>
	 * Warning: the input 'list' is modified.
	 */
	public CircularDLinkedList(DLinkedList<X> list) {
		this.first = new DListNode<X>(null, null, null);
		this.first.next = list.getFirst(); // the two lists start from the same node
		
		DListNode<X> start=list.getFirst();
		DListNode<X> end=list.getLast();
		
		start.previous=end; // the original linked list is made circular
		end.next=start;
		
		this.size = list.size();
	}
	
	/** 
	 * Create a clone (a copy) of a circular doubly linked list. <br>
	 * <br>
	 * Remark: the two lists shared the elements, but the nodes are different. <br>
	 * The original (current) list is not modified.
	 */
	public CircularDLinkedList<X> clone() {
		if(this.size()==0)
			throw new Error("Warning: not supported operation on empty lists");
		
		CircularDLinkedList<X> result;
		
		// first create a (not circular) doubly linked list
		DLinkedList<X> list=new DLinkedList<X>(); // empty new list
		DListNode<X> firstNode=this.getFirst();
		X e=firstNode.getElement(); // add first element
		list.addLast(e);
		
		int count=1; // only for debugging
		DListNode<X> n=firstNode.getNext();
		while(n!=firstNode) { // add all elements to the new list
			e=n.getElement();
			list.addLast(e);
			n=n.getNext();
			count++; // only for debugging
		}
		
		result=new CircularDLinkedList<X>(list); // make the 'list' circular
		
		if(this.size()!=result.size()) // only for debugging
			throw new Error("Error: the size of the original and clone lists do not coincide");
		
		return result;
	}
	
	public int size() { 
		return this.size; 
	}
	
	/**
	 * Check whether the list is empty
	 */
	public boolean isEmpty() {
		return this.size == 0; 
	}

	/**
	 * Return the first node of the list
	 * The result is a null reference if the list is empty
	 */
	public DListNode<X> getFirst() {
		return this.first.getNext();
	}

	/**
	 * Insert a new element after a given node in the list
	 * @param el the element to insert
	 * @param nPnode the place where the element will be inserted
	 */
	public void insertAfter(DListNode<X> pNode, X el) {
		DListNode<X> item = new DListNode<X>(el, pNode.next, pNode);
		item.previous.next = item;
		item.next.previous = item;
		size++;
	}

	/**
	 * Insert a new element before a given node in the list
	 * @param el the element to insert
	 * @param nPnode the place where the element will be inserted
	 */
	public void insertBefore(DListNode<X> pNode, X el) {
		DListNode<X> item = new DListNode<X>(el, pNode, pNode.previous);
		item.previous.next = item;
		item.next.previous = item;
		size++;
	}
	
	/**
	 * Remove a given node from the list
	 * @param nPnode the node to delete
	 */
	public void delete(DListNode<X> pNode) {
		if(pNode==null)
			return;
		if(this.first.next!=pNode) {
			pNode.next.previous = pNode.previous;
			pNode.previous.next = pNode.next;
		}
		else {
			this.first.next=pNode.next;
			pNode.next.previous = pNode.previous;
			pNode.previous.next = pNode.next;
			//System.out.println("Warning: removing first element in a circuar list");
		}
		pNode.setElement(null); // for debugging
		if(first.next==null || first.next.getElement()==null)
			throw new Error("Error: wrong edge deletion in cycle");
		size--;
	}

	/**
	 * Return a String representing the list
	 */
	public String toString() {		
		if(this.isEmpty())
			return "empty list";
		String result="[\n";
		DListNode<X> firstNode=this.getFirst();
		result=result+firstNode.getElement()+", \n";
		
		DListNode<X> n=firstNode.getNext();
		int count=0; // only for debugging
		while(n!=firstNode) {
			result=result+n.getElement()+", \n";
			n=n.getNext();
			count++;
		}
		return result+"]";
	}

	/**
	 * Check whether the circular list contains a given node (check reference equality) <br>
	 * <br>
	 * Remark: useful for debugging
	 */
	public boolean containsNode(DListNode<X> node) {		
		if(this.isEmpty())
			return false;
		DListNode<X> firstNode=this.getFirst();
		if(firstNode==node)
			return true;
		
		DListNode<X> n=firstNode.getNext();
		while(n!=firstNode) {
			if(n==node)
				return true;
			n=n.getNext();
		}
		return false;
	}
	
	// for testing the correctness of methods
	public static void main(String[] args) {
		System.out.println("Testing (circular) doubly linked list");
		DLinkedList<Integer> l=new DLinkedList<Integer>();
		l.add(1);
		l.add(2);
		l.add(3);
		l.add(4);
		System.out.println("l: "+l);
		CircularDLinkedList<Integer> c=new CircularDLinkedList<Integer>(l); // make the first list circular
		System.out.println("original list c: "+c);
		
		CircularDLinkedList<Integer> clone=c.clone();
		c.insertAfter(c.getFirst(), 10);
		System.out.println("c after insertion: "+c);
		
		System.out.println("clone: "+clone);
	}

}
