package Jcg.util;

/** 
 * An implementation of a bit-vector (using memory words on 64 bits)
 * @author Luca Castelli Aleardi (older version by Yonik)
 */
public class BitSet64 implements BitSetInterface {
	
	private int numBits; // length of the bit-vector (number of bits)
	private long[] bits; // array storing the data
	private int wlen; // number of words (elements) used in the array

	/** 
	 * Constructs an OpenBitSet large enough to hold numBits.
	 * @param numBits
	 */
	public BitSet64(int numBits) {
		this.numBits=numBits;
		this.bits = new long[bits2words(numBits)];
		this.wlen = bits.length;
	}

	/** 
	 * Constructs a BitSet64 from an input array of booleans
	 * @param A  	an array of booleans
	 */
	public BitSet64(boolean[] A) {
		this.numBits=A.length;
		this.bits = new long[bits2words(numBits)];
		this.wlen = bits.length;
		
		for(int i=0;i<this.numBits;i++)
			if(A[i]==true)
				this.set(i);
	}

	/**
	 * Returns the size of the bitset (number of words)
	 */
	public int size() {
		return this.wlen;
	}
	
	/**
	 * Returns the size of the bitset (number of words)
	 */
	public int bitSize() {
		return this.numBits;
	}

	/** Expert: returns the long[] storing the bits */
	public long[] getBits() {
		return bits;
	}

	/** 
	 * Returns true or false for the specified bit index.
	 * The index should be less than the OpenBitSet size
	 */
	public boolean get(int index) {
		int i = index >> 6; // div 64
		// signed shift will keep a negative index and force an
		// array-index-out-of-bounds-exception, removing the need for an explicit check.
		int bit = index & 0x3f; // mod 64
		long bitmask = 1L << bit;
		return (bits[i] & bitmask) != 0;
	}

	/** 
	 * Returns 1 if the bit is set, 0 if not.
	 * The index should be less than the OpenBitSet size
	 */
	public int getBit(int index) {
		int i = index >> 6; // div 64
		int bit = index & 0x3f; // mod 64
		return ((int) (bits[i] >>> bit)) & 0x01;
	}

/*
public boolean get2(int index) {
int word = index >> 6;            // div 64
int bit = index & 0x0000003f;     // mod 64
return (bits[word] << bit) < 0;   // hmmm, this would work if bit order were reversed
// we could right shift and check for parity bit, if it was available to us.
}
*/

	/** Sets the bit at the specified index.
	 * The index should be less than the OpenBitSet size.
	 */
	public void set(int index) {
		int wordNum = index >> 6; // div 64
		int bit = index & 0x3f; // mod 64
		 long bitmask = 1L << bit;
		 bits[wordNum] |= bitmask;
	}

	/** 
	 * Clears a bit.
	 * The index should be less than the OpenBitSet size.
	 */
	public void clear(int index) {
		int wordNum = index >> 6;
		int bit = index & 0x03f;
		long bitmask = 1L << bit;
		bits[wordNum] &= ~bitmask;
		// hmmm, it takes one more instruction to clear than it does to set... any
		// way to work around this?  If there were only 63 bits per word, we could
		// use a right shift of 10111111...111 in binary to position the 0 in the
		// correct place (using sign extension).
		// Could also use Long.rotateRight() or rotateLeft() *if* they were converted
		// by the JVM into a native instruction.
		// bits[word] &= Long.rotateLeft(0xfffffffe,bit);
	}

	/** returns the number of 64 bit words it would take to hold numBits */
	public static int bits2words(int numBits) {
		return (int) (((numBits - 1) >>> 6) + 1);
	}

	/** return an array of booleans corresponding to the bit-vector */
	public boolean[] toBoolean() {
		boolean[] result=new boolean[this.numBits];
		
		for(int i=0;i<this.numBits;i++) {
				result[i]=this.get(i);
		}
		return result;
	}

	/** 
	 * Convert the BitSet64 into an array of booleans
	 * @param  	array  	an input array of booleans 
	 **/
	public void toBoolean(boolean[] array) {
		if(this.numBits!=array.length)
			throw new Error("The length of the array is wrong: "+array.length+", vs. "+this.numBits);
		
		for(int i=0;i<this.numBits;i++) {
				array[i]=this.get(i);
		}
	}

	public String toString() {
		String result="";
		for(int i=0;i<this.numBits;i++) {
			result=result+this.getBit(i);
		}
		return result;
	}


}
