package Jcg.polyhedron;

import java.util.*;

import Jcg.geometry.*;
import Jcg.util.PrintUtil;


/**
 * Pointer-based implementation of the <em>Half-edge</em> data structure for polyhedral meshes
 * (for representing planar and surface orientable meshes). <br>
 * <br>
 * This data structure allows us to represent both planar maps (embedded in 2D) and 3D surface meshes (embedded in R^3). <br>
 * The generic type <code>X</code> allows us to choose between the two cases: <code>X</code> may be set either as <code>Point_2</code> 
 * or <code>Point_3</code>
 *
 * @author Luca Castelli Aleardi (Ecole Polytechnique, INF562/INF574, 2010-2022)
 *
 * @see Primitives for the Manipulation of an orientable surface
 * 
 * @param X  	the type of geometric coordinates (2D or 3D), real or integer coordinates
 */
public class Polyhedron_3<X extends Point_>{
	public static final int NOTHING=0, ALL=1;
	
	public int verbosity=NOTHING;

	/** dynamic array storing the vertices of the mesh */
    public ArrayList<Vertex<X>> vertices;
    /** dynamic array storing the faces of the mesh */
    public ArrayList<Face<X>> facets;
    /** dynamic array storing the half-edges of the mesh */
    public ArrayList<Halfedge<X>> halfedges;

    public Polyhedron_3() {
        vertices=new ArrayList<Vertex<X>>();
        facets=new ArrayList<Face<X>>();
        halfedges=new ArrayList<Halfedge<X>>();
    }
   
    /**
     * Initialize a polyhedron wiith a given number of vertices, edges and faces
     * @param n  number of vertices
     * @param e  number of half-edges
     * @param f  number of faces
     */
    public Polyhedron_3(int n, int e, int f) {
        vertices=new ArrayList<Vertex<X>>(n);
        facets=new ArrayList<Face<X>>(f);
        halfedges=new ArrayList<Halfedge<X>>(e);
    }

    /**
     * Remove all undefined (<code>null</code> references) vertices, half-edges and faces. <br>
     * Indices of vertices/edges/faces are reset.
     */
    public void clean() {
    	if(this.verbosity==ALL)
    		System.out.print("Removing undefined references...");
    	int n=0, e=0, f=0; // number of undefined references
    	int N, E, F; // number of vertices/edges/faces (defined)

    	// first check whether there are undefined elements
    	for(Vertex<X> v: this.vertices)
    		if(v==null)
    			n++;
    	for(Halfedge<X> h: this.halfedges)
    		if(h==null)
    			e++;
    	for(Face<X> face: this.facets)
    		if(face==null)
    			f++;

    	if(n==0 && e==0 && f==0) { // nothing to do: no undefined elements (no 'null' references)
    		if(this.verbosity==ALL)
    			System.out.println("done (no references undefined to be removed)");
    		return;
    	}

    	// update the dynamic arrays
    	String msg="removed references: ";
    	if(n>0) {
    		ArrayList<Vertex<X>> newVertices=new ArrayList<Vertex<X>>(n);
    		for(Vertex<X> v: this.vertices)
    			if(v!=null)
    				newVertices.add(v);

    		this.vertices=newVertices;
    		msg=msg+n+" vertices, ";
    	}
    	if(e>0) {
    		ArrayList<Halfedge<X>> newHalfedges=new ArrayList<Halfedge<X>>(e);
    		for(Halfedge<X> h: this.halfedges)
    			if(h!=null)
    				newHalfedges.add(h);

    		this.halfedges=newHalfedges;
    		msg=msg+e+" halfedges, ";
    	}
    	if(f>0) {
    		ArrayList<Face<X>> newFacets=new ArrayList<Face<X>>(f);
    		for(Face<X> face: this.facets)
    			if(face!=null)
    				newFacets.add(face);

    		this.facets=newFacets;
    		msg=msg+f+" faces";
    	}

    	this.resetMeshIndices(); // vertex/face/edge indices are recomputed
    	if(this.verbosity==ALL)
    		System.out.println("done ("+msg+")");
    }
    
    class ColorDecorator extends Decorator<Vertex<X>,Integer> {
    }
    
    public void DecorateVertices() {
    	ColorDecorator color=new ColorDecorator();
    	color.setDecoration(vertices.get(0),0);
    }

    public int sizeOfVertices() {
    	return this.vertices.size();
    }
    
    public int sizeOfFacets() {
    	return this.facets.size();
    }
    public int sizeOfHalfedges() {
    	return this.halfedges.size();
    }
    
    /**
     * compute the degree of a vertex
     */
    public int vertexDegree(Vertex<X> v) {
    	int result=0;
    	Halfedge<X> e=v.getHalfedge();
    	
    	Halfedge<X> pEdge=e.getNext().getOpposite();
    	while(pEdge!=e) {
    		pEdge=pEdge.getNext().getOpposite();
    		result++;
    	}
    	
    	return result+1;
    }
    
    /**
     * Check whether the mesh is closed (no boundaries)
     * 
     * @return true  if there are no border edges
     */
	public boolean isClosed() {
		for(Halfedge<X> h: this.halfedges) {
			if(h!=null && h.face==null)
				return false;
		}
		return true;
	}

	/**
	 * Check whether all vertices have degree exactly two. <br>
	 * 
	 * <b>Warning</b>: not implemented yet
	 * 
	 * @return true if all vertices have exactly two incident edges
	 */	
	public boolean isPureBivalent() {
		throw new Error("To be completed");
	}

	/**
	 * Check whether all vertices have degree exactly three. <br>
	 * <b>Warning</b>: not implemented yet
	 * 
	 * @return true if all vertices have exactly three incident edges
	 */	
	public boolean isPureTrivalent() {
		throw new Error("to be completed");
	}

	/**
	 * Check whether the polyhedral surface is a triangle mesh (all faces are triangles)
	 * 
	 * @return true  iff the every face is a triangle
	 */	
	public boolean isPureTriangle() {
		for(Face<X> f: this.facets) {
			if(f!=null) {
				Halfedge<X> h=f.halfedge;
				if(h.next.next.next!=h)
					return false;
			}
		}
		return true;
	}
	
	/**
	 * Check whether the polyhedral surface is a <em>quad mesh</em> (all faces are quadrangles)
	 * 
	 * @return true  iff the every face is a quadrangle
	 */	
	public boolean isPureQuad() {
		for(Face<X> f: this.facets) {
			if(f!=null) {
				Halfedge<X> h=f.halfedge;
				if(h.next.next==h) // degree two face (h is multiple edge)
					return false;
				if(h.next.next.next.next!=h)
					return false;
			}
		}
		return true;
	}

	/**
	 * <b>Warning</b>: not implemented yet
	 * 
	 * @param h  half-edge
	 * @return true iff the connected component denoted by <code>h</code> is a triangle
	 */	
	public boolean isTriangle(Halfedge<X> h) {
		throw new Error("To be completed");
	}

	/**
	 * Return the genus of the mesh (given by Euler formula). <br>
	 * <b>Warning</b>: works only for meshes with one connected component
	 * 
	 * @return  the genus of the mesh
	 */
	public int genus() {
		int n=sizeOfVertices();
		int e=sizeOfHalfedges();
		int f=sizeOfFacets();
		int b=this.numberOfBoundaries();
		return -(n-e/2+f-2+b)/2;
	}

    /**
     * Compute and return the number of boundaries
     */
    public int numberOfBoundaries() {
		//System.out.print("Processing boundaries of halfedge representation ...");
    	// store boundary edges (corresponding to half-edges without opposite reference)
    	HashSet<Halfedge<X>> isTreated=new HashSet<Halfedge<X>>(); // says whether a boundary edges has already been treated
    	LinkedList<Halfedge<X>> boundaryEdges=new LinkedList<Halfedge<X>>(); // stores boundary edges
        int nInsertedHalfedges=0; // counts the number of inserted halfedges
        
        for(Halfedge<X> e: this.halfedges) { // store all boundary half-edges
        	if(e!=null && e.face==null)
        		boundaryEdges.add(e);
        }
        
        int bCount=0; // counts the number of boundaries
        while (boundaryEdges.isEmpty()==false) { // set opposite references of all boundary edges
        	Halfedge<X> firstEdge=boundaryEdges.poll(); // get a boundary
        	//System.out.println("processing edge "+printEdge(firstEdge));
        	if(isTreated.contains(firstEdge)==false) { // the edge belongs to a new boundary
        		isTreated.add(firstEdge); // mark the half-edge as treated
        		Halfedge<X> e=firstEdge.getNext();
        		while(e!=firstEdge) { // edge e turns around the boundary face
        			isTreated.add(e); // mark the half-edge as treated
        			e=e.getNext(); // visit next boundary half-edge
        		}
        		bCount++;
        	}
        }
        return bCount;
    }

    /**
     * Check whether the polyhedral surface is combinatorially consistent: manifold polygonal mesh.<br>
     *  <br>
     * If <code>borders==true</code> normalization of the border edges is checked too. <br>
     * 
     * This method checks that each facet is at least a triangle and that 
     * the two incident facets of a non-border edge are distinct. 
     */	    
    public boolean isValid(boolean borders) {
    	this.clean(); // LCA: added january 2023, CgShop2023
    	int undefinedEdges=0, undefinedFaces=0, undefinedVertices=0;
    	boolean valid=true;
        System.out.print("Checking Polyhedron...");
        int n=this.vertices.size();
        int e=this.halfedges.size();
        int f=this.facets.size();

        int i=0;
        for(Halfedge<X> pedge: this.halfedges) {
        	if(pedge!=null) {
        		if(pedge.getNext()==null || pedge.next.prev!=pedge) { 
        			System.out.println(PrintUtil.ANSI_RED+"error next_edge: "+i+PrintUtil.ANSI_RESET); 
        			valid=false; 
        		}
        		if(pedge.getPrev()==null || pedge.prev.next!=pedge) { 
        			System.out.println(PrintUtil.ANSI_RED+"error prev_edge: "+i+PrintUtil.ANSI_RESET); 
        			valid=false; 
        		}
        		if(pedge.getVertex()==null) { 
        			System.out.println(PrintUtil.ANSI_RED+"error vertex: "+i+PrintUtil.ANSI_RESET); 
        			valid=false;
        		}
        		if(pedge.opposite==null || pedge.opposite.opposite==null || pedge.opposite.opposite!=pedge) {
        			System.out.println(PrintUtil.ANSI_RED+"error opposite edge: "+i+PrintUtil.ANSI_RESET); 
        			valid=false;        		
        		}
        		if(pedge.face!=null && pedge.face==pedge.getOpposite().face) { 
        			System.out.print(PrintUtil.ANSI_MAGENTA+"warning multiple edge: ");
        			System.out.println(" halfedge "+i+" is incident twice to face "+pedge.face+PrintUtil.ANSI_RESET);
        			valid=false; 
        		}
        	}
        	else {
        		undefinedEdges++;
        	}
        	i++;
        }            
        
        i=0;
        for(Face<X> pface: this.facets){
        	if(pface!=null) { 
        		if(pface.halfedge==null) { 
        			System.out.println(PrintUtil.ANSI_RED+"Error: face.halfedge NULL for face f"+pface.index+PrintUtil.ANSI_RESET); 
        			valid=false; break; 
        		}
        		if(pface.degree()<3) { 
        			System.out.println(PrintUtil.ANSI_MAGENTA+"Warning: face of degree less than 3, f"+i+PrintUtil.ANSI_RESET); 
        			valid=false; 
        		}
        		if(pface.halfedge.face!=pface) { System.out.println(PrintUtil.ANSI_RED+"error face.halfedge"+PrintUtil.ANSI_RESET); 
        		valid=false; 
        		}
        	}
        	else {
        		undefinedFaces++;
        	}
        	i++;
        }
        
        i=0;
        for(Vertex<X> pvertex: this.vertices){
        	if(pvertex!=null) { 
        		if(pvertex.halfedge==null) { 
        			System.out.println(PrintUtil.ANSI_RED+"error vertex.halfedge: v"+i+PrintUtil.ANSI_RESET); 
        			valid=false; 
        		}
        		if(pvertex.getPoint()==null) { 
        			System.out.println(PrintUtil.ANSI_RED+"error vertex.point not defined: v"+i+PrintUtil.ANSI_RESET); 
        			valid=false; 
        		}
        		if(pvertex.halfedge.vertex!=pvertex) { System.out.println("error vertex.halfedge: v"+i); valid=false; }
        	}
        	else
        		undefinedVertices++;
        	i++;
        }            

        if(valid==true) {
        	System.out.println("ok\t");
        	
        	int b=this.numberOfBoundaries();
        	int g=-(n-e/2+f-2+b)/2;
        	System.out.print("n: "+n+"  e: "+e/2+"  f: "+f+"  b:"+b);
        	System.out.println("  genus: "+g);

        	System.out.print("The mesh ");
        	if(this.isClosed())
        		System.out.print("is closed (no boundaries)");
        	else
        		System.out.print("is open (has boundaries)");
        	if(this.isPureTriangle())
        		System.out.println(" and pure triangle");
        	else if(this.isPureQuad())
        		System.out.println(" and pure quad");
        	else
        		System.out.println(" and polygonal");

        	String msg="";
        	if(undefinedFaces>0)
        		msg=msg+"undefined faces";
        	if(undefinedEdges>0)
        		msg=msg+"undefined edges";
        	if(undefinedVertices>0)
        		msg=msg+"undefined vertices";
        	
        	if(msg.length()>0)
        		System.out.println(msg);
        }
        else
         	System.out.println("not valid"); 
        
        return valid;
    }

    /**
     * Check whether three half-edges define a triangle face of the mesh.
     * The three input half-edges are assumed to have a coherent orientation. <br>
     * <br>
     * <b>Warning</b>: it works only for triangulations
     */	    
    public boolean isFace(Halfedge<X> e1, Halfedge<X> e2, Halfedge<X> e3) {
    	if(e1.getFace()==e2.getFace() && e2.getFace()==e3.getFace()) // they are 'inside' the same face
    		return true;
    	if(e1.getOpposite().getFace()==e2.getOpposite().getFace() && e2.getOpposite().getFace()==e3.getOpposite().getFace())
    		return true;
    	
    	return false;
    }

    /**
     * Check whether three edges define an (oriented) cycle of length 3 (a triangle, not necessarily a face). <br>
     * The three input half-edges are assumed to have a coherent orientation <br>
     * <br>
     * <b>Warning</b>: it works only for (closed) triangulations
     */	    
    public boolean isCycle(Halfedge<X> e1, Halfedge<X> e2, Halfedge<X> e3) {
    	if(e1.getVertex()!=e2.getOpposite().getVertex())
    		return false;
    	if(e2.getVertex()!=e3.getOpposite().getVertex())
    		return false;
    	if(e3.getVertex()!=e1.getOpposite().getVertex())
    		return false;
    	
    	return true;
    }

    /**
     * Check whether three edges define an (oriented) cycle of length 3 (a triangle, not necessarily a face). <br>
     * The three input halfedges are assumed to have a coherent orientation <br>
     * <br>
     * <b>Warning</b>: it works only for (closed) triangulations
     */	    
    public boolean isSeparatingCycle(Halfedge<X> e1, Halfedge<X> e2, Halfedge<X> e3) {
    	// first check that is a cycle
    	if(e1.getVertex()!=e2.getOpposite().getVertex())
    		return false;
    	if(e2.getVertex()!=e3.getOpposite().getVertex())
    		return false;
    	if(e3.getVertex()!=e1.getOpposite().getVertex())
    		return false;
    	
    	if(this.isFace(e1, e2, e3))
    		return false;
    	
    	return true;
    }

    /**
     * Return all separating separating containing the half-edge 'e1'. All resulting triangles are oriented.
     * A triangle is stored using an array of 3 hal-edges. <br>
     * <br>
     * <b>Warning</b>: it works only for (closed) triangulations
     * 
     * @return  the list of all (oriented) separating triangles 
     */	    
    public ArrayList<Halfedge<X>[]> getSeparatingCycles(Halfedge<X> e) {
    	if(e==null)
    		return null;
    	
    	int countFaces=0; // just for debug
    	int countSeparatingTriangles=0;
    	
    	ArrayList<Halfedge<X>[]> result=new ArrayList<Halfedge<X>[]>();
    	
    	Vertex<X> u=e.getOpposite().getVertex(); // source vertex
    	Vertex<X> v=e.getVertex(); // destination vertex
    	
    	List<Halfedge<X>> neighborsU=u.getOutgoingHalfedges();
    	List<Halfedge<X>> neighborsV=v.getOutgoingHalfedges();
    	
    	for(Halfedge<X> g: neighborsV) {
    		if(e!=g && e.getOpposite()!=g) {
    			for(Halfedge<X> h: neighborsU) {
    				if(e!=h && e.getOpposite()!=h && g!=h && g!=h.getOpposite()) {
    					//System.out.print("\t ("+u.index+", "+v.index+")");
    					//System.out.print("\t ("+g.getOpposite().getVertex().index+", "+g.getVertex().index+")");
    					//System.out.print("\t ("+h.getVertex().index+", "+h.getOpposite().getVertex().index+")");
    					// check whether the edges 'e', 'g', and 'h.opposite' define an (oriented and valid) face
    					if(this.isFace(e, g, h.getOpposite())==true) {// remark: remind that 'h' is outgoing from the source 'u'
    						//System.out.print("\t face");
    						countFaces++;
    					}
    					else {
    						//System.out.print("\t not face");
    						if(this.isSeparatingCycle(e, g, h.getOpposite())) {
    							Halfedge<X>[] t=new Halfedge[3];
    							t[0]=e;
    							t[1]=g;
    							t[2]=h.getOpposite();
    							result.add(t);
    							countSeparatingTriangles++;
    						}
    					}
    					//System.out.println();"e ("+u.index+","+v.index+") faces: "+countFaces+
    				}
    			}
    		}
    	}
    	
    	if(countFaces!=2)
    		throw new Error("Error: edge "+e.index+" ("+u.index+", "+v.index+") is non incident to 2 faces");
    	
    	return result;
    }

    /**
     * Return all separating separating in the triangulation <br>
     * <br>
     * Warnings: <br> 
     * -) it assumes that all edges and vertices have an index <br>
     * -) it works only for (closed) triangulations
     * 
     * @return  the list of all (oriented) separating triangles 
     */	    
    public ArrayList<Halfedge<X>[]> getSeparatingTriangles() {
    	ArrayList<Halfedge<X>[]> result=new ArrayList<Halfedge<X>[]>();
    	
    	int countSeparatingTriangles=0;
    	
    	ArrayList<Halfedge<X>[]> triangles;
    	HashSet<SeparatingTriangle> set=new HashSet<SeparatingTriangle>();
    	
    	for(Halfedge<X> e: this.halfedges) {
    		if(e!=null && e.getOpposite().getVertex().index<e.getVertex().index) { // process each edge only once
    			triangles=this.getSeparatingCycles(e); // separating triangles containing the edge 'e'
    			countSeparatingTriangles=countSeparatingTriangles+triangles.size();
    			
    			for(Halfedge<X>[] t:triangles) { // iterate over all triangles
    				 // add only one copy of the triangle
    				SeparatingTriangle sep=new SeparatingTriangle(t);
    				if(set.contains(sep)==false) { // add if the first edge has minimal index
    					result.add(t);
    					set.add(sep);
    				}
    			}
    		}
    	}
    	
    	if(countSeparatingTriangles%3!=0) // check validity: each triangle should be represented three times
    		throw new Error("Error: wrong number of separating triangles with repetions: "+countSeparatingTriangles);
    	if(countSeparatingTriangles/3!=result.size()) // check validity: each triangle is stored only once
    		throw new Error("Error: wrong number of separating triangles "+result.size()+", "+countSeparatingTriangles);

    	//System.out.println("separating triangles: "+result.size());
    	return result;
    }
    
    /**
     * Perform the flip of an edge <tt>e=(u, v)</tt> <br>
     * Remarks: <br>
     * -) an edge can be flipped only if the two incident faces <tt>(u, v, w)</tt> and <tt>(u, v, z)</tt> are both triangles <br>
     * -) boundary edges cannot be flipped <br>
     * -) <tt>w</tt> and <tt>z</tt> must be not adjacent: otherwise we create a multiple edge
     * 
     * @param e  the half-edge to be flipped
     */
    public void flipEdge(Halfedge<X> e) {
    	if(e==null)
    		throw new Error("Error: null edge");
    	if(e.getFace()==null || e.getOpposite().getFace()==null) // it cannot flip boundary edges
    		return;
    	
    	Vertex<X> v=e.getVertex();
    	Vertex<X> u=e.getOpposite().getVertex();
    	Vertex<X> w=e.getNext().getVertex();
    	Vertex<X> z=e.getOpposite().getNext().getVertex();
    	Face<X> f1=e.getFace();
    	Face<X> f2=e.getOpposite().getFace();
    	
    	if(f1.isTriangle()==false || f2.isTriangle()==false) // flips only concern triangle faces
    		return;
    	// before flipping we must check that 'w' and 'z' are not adjacent, to preserve combinatorial validity
    	List<Halfedge<X>> neighbors=w.getOutgoingHalfedges();
    	for(Halfedge<X> neighbor: neighbors) {
    		if(neighbor.getVertex()==z)
    			return;
    	}
    	
    	Halfedge<X> eVW=e.getNext(); // edge (V, W) oriented from V to W
    	Halfedge<X> eWU=e.getPrev();
    	Halfedge<X> eZV=e.getOpposite().getPrev(); // edge (Z, V)
    	Halfedge<X> eUZ=e.getOpposite().getNext();
    	
    	
    	// update vertices
    	u.setEdge(eWU);
    	v.setEdge(eZV);
    	w.setEdge(eVW);
    	z.setEdge(eUZ);
    	
    	// update the destination vertices of 'e' and its opposite half-edge
    	e.setVertex(w);
    	e.getOpposite().setVertex(z);
    	
    	eUZ.setFace(f1);
    	eVW.setFace(f2);
    	f1.setEdge(e);
    	f2.setEdge(e.getOpposite());
    	
    	// update the three halfedges inside F1
    	e.setNext(eWU);
    	eWU.setPrev(e);
    	eWU.setNext(eUZ);
    	eUZ.setPrev(eWU);
    	eUZ.setNext(e);
    	e.setPrev(eUZ);
    	
    	e.getOpposite().setNext(eZV);
    	eZV.setPrev(e.getOpposite());
    	eZV.setNext(eVW);
    	eVW.setPrev(eZV);
    	eVW.setNext(e.getOpposite());
    	e.getOpposite().setPrev(eVW);
    	
    	System.out.println("Edge e"+e.index+" flipped");
    }
    
    /**
     * Split a face by inserting a new vertex <p> <br>
     * <b>Warning</b>: it works only for triangular faces<p>
     * 
     * Remark: faces are assumed to be ccw oriented (for combinatorics)
     */
    public void createCenterVertex(Face<X> f, X point){
    	int degree=f.degree();
    	if(degree!=3)
    		throw new Error("Error: the current implementation only works for triangle faces");
    	
    	// retrieve the three edges of the face
    	Halfedge<X> e=f.getEdge();
    	Halfedge<X> e1=e.getNext();
    	Halfedge<X> e2=e1.getNext();
    	
    	// retrieve the three vertices of the face
    	Vertex<X> v=e.getVertex(); // destination vertex of the edge 'e'
    	Vertex<X> u=e2.getVertex(); // source vertex of the edge'e'
    	Vertex<X> w=e1.getVertex();
    	
    	// create a new vertex
    	Vertex<X> newVertex=new Vertex<X>(point);
    	newVertex.index=this.sizeOfVertices();
    	this.vertices.add(newVertex); // the new vertex is added at the end of the list of vertices
    	
    	// create two new faces
    	Face<X> f1=new Face<X>();
    	f1.index=this.sizeOfFacets();
    	this.facets.add(f1); // the new face is added at the end of the list of faces
    	Face<X> f2=new Face<X>();
    	f2.index=this.sizeOfFacets();
    	this.facets.add(f2); // the new face is added at the end of the list of faces

    	// create 3 new pairs of halfedges
    	Halfedge<X> eUOut=new Halfedge<X>(); // halfedge outgoing from u
    	Halfedge<X> eUIn=new Halfedge<X>(); // halfedge oriented toward u
    	Halfedge<X> eVOut=new Halfedge<X>(); // halfedge outgoing from v
    	Halfedge<X> eVIn=new Halfedge<X>(); // halfedge oriented toward v
    	Halfedge<X> eWOut=new Halfedge<X>(); // halfedge outgoing from w
    	Halfedge<X> eWIn=new Halfedge<X>(); // halfedge oriented toward w
    	// add the new 6 halfedges to the mesh (setting their indices)
    	eUOut.index=this.sizeOfHalfedges();
    	this.halfedges.add(eUOut);
    	eUIn.index=this.sizeOfHalfedges();
    	this.halfedges.add(eUIn);
    	eVOut.index=this.sizeOfHalfedges();
    	this.halfedges.add(eVOut);
    	eVIn.index=this.sizeOfHalfedges();
    	this.halfedges.add(eVIn);
    	eWOut.index=this.sizeOfHalfedges();
    	this.halfedges.add(eWOut);
    	eWIn.index=this.sizeOfHalfedges();
    	this.halfedges.add(eWIn);
    	
    	// set references between faces and incident halfedges
    	f.halfedge=e;
    	f1.halfedge=e1;
    	f2.halfedge=e2;
    	e.face=f;
    	e1.face=f1;
    	e2.face=f2;
    	// set the two edges incident to the face f
    	eUIn.face=f;
    	eVOut.face=f;
    	// set the two edges incident to the face f1
    	eVIn.face=f1;
    	eWOut.face=f1;
    	// set the two edges incident to the face f2
    	eWIn.face=f2;
    	eUOut.face=f2;
    	
    	// set references between faces and incident vertices
    	newVertex.halfedge=eUOut;
    	eUIn.vertex=u;
    	eUOut.vertex=newVertex;
    	// set the two edges incident to the face f1
    	eVIn.vertex=v;
    	eVOut.vertex=newVertex;
    	// set the two edges incident to the face f2
    	eWIn.vertex=w;
    	eWOut.vertex=newVertex;
    	
    	// set 'prev' and 'next' references between halfedges, for the face 'f'
    	e.next=eVOut;
    	eVOut.prev=e;
    	e.prev=eUIn;
    	eUIn.next=e;
    	eUIn.prev=eVOut;
    	eVOut.next=eUIn;

    	// set 'prev' and 'next' references between halfedges, for the face 'f1'
    	e1.next=eWOut;
    	eWOut.prev=e1;
    	e1.prev=eVIn;
    	eVIn.next=e1;
    	eVIn.prev=eWOut;
    	eWOut.next=eVIn;
    	
    	// set 'prev' and 'next' references between halfedges, for the face 'f2'
    	e2.next=eUOut;
    	eUOut.prev=e2;
    	e2.prev=eWIn;
    	eWIn.next=e2;
    	eWIn.prev=eUOut;
    	eUOut.next=eWIn;
    	
    	// set 'opposite' references between 3 new pairs of halfedges
    	eVOut.opposite=eVIn;
    	eVIn.opposite=eVOut;
    	eWOut.opposite=eWIn;
    	eWIn.opposite=eWOut;
    	eUOut.opposite=eUIn;
    	eUIn.opposite=eUOut;
    }

    /**
     * Replace the star of a vertex v by a new face. <br>
     * 
     * <b>Warning</b>: to be completed, not implemented yet
     * 
     * @param v  the vertex to be removed
     */	       
    public void eraseCenterVertex(Vertex<X> v) {
    	throw new Error("To be completed");
    }

    /**
     * Erase the vertex 'v' pointed by 'e', by replacing the star of 'v' by a new face. <br>
     * <br>
     * <b>Warning</b>: this method is not computationally efficient, requiring O(n) time: 
     * it remove all elements (6 halfedges, 1 vertex, 2 faces) from the corresponding collections ('ArrayList'). <p>
     * <p>
     * Remarks: <p>
     * -) it only does work with vertices having degree 3 <br>
     * -) the faces incident to vertex 'v' are assumed to exist (no holes incident to 'v') <br>
     * -) faces are assumed to be ccw oriented (only for combinatorics)
     * 
     * @return the halfedge <tt>h</tt> which precedes 'e' in ccw direction (in the new triangular face)
     */	       
    public Halfedge eraseCenterVertex(Halfedge<X> e) {
    	Vertex v=e.getVertex();
    	if(this.vertexDegree(v)!=3) {
    		throw new Error("Error: the current implementation only works for vertices of degree 3");
    	}
    	
    	// get all 6 half-edges incident to vertex 'v' (they must be removed)
    	Halfedge<X> e1=e;
    	Halfedge g1=e.getOpposite();
    	Halfedge e2=g1.getPrev();
    	Halfedge g2=e2.getOpposite();
    	Halfedge e3=g2.getPrev();
    	Halfedge g3=e3.getOpposite();
    	
    	// get the 3 faces incident to vertex 'v' (f1 must be updated, f2 and f3 must be removed)
    	Face f1=e1.getFace();
    	Face f2=e2.getFace();
    	Face f3=e3.getFace();
    	if(f1==null || f2==null || f3==null)
    		throw new Error("Error: one of the incident faces is null (hole in the mesh)");
    	
    	// get the three vertices neighbors of 'v' (they must be updated)
    	Vertex v1=g1.getVertex();
    	Vertex v2=g2.getVertex();
    	Vertex v3=g3.getVertex();
    	
    	// get the three halfedges defining the new triangular face (v1, v2, v3) which must be "updated"
    	Halfedge h1=e1.getPrev();
    	Halfedge h2=e2.getPrev();
    	Halfedge h3=e3.getPrev();
    	
    	// update the references involving halfedges h1, h2 and h3
    	h1.next=h2;
    	h2.prev=h1;
    	h2.next=h3;
    	h3.prev=h2;
    	h3.next=h1;
    	h1.prev=h3;
    	
    	// h1, h2 and h3 are all incident to the "new" face f1
    	h1.face=f1;
    	h2.face=f1;
    	h3.face=f1;
    	
    	// vertices v1, v2 and v3 must be pointed by h1, h2 and h3
    	v1.halfedge=h1;
    	v2.halfedge=h2;
    	v3.halfedge=h3;
    	
    	// the face f1 must be incident to the halfedge h1
    	f1.halfedge=h1;
    	
    	// remove all elements
    	this.facets.remove(f2);
    	this.facets.remove(f3);
    	this.halfedges.remove(e1);
    	this.halfedges.remove(e2);
    	this.halfedges.remove(e3);
    	this.halfedges.remove(g1);
    	this.halfedges.remove(g2);
    	this.halfedges.remove(g3);
    	this.vertices.remove(v);
    	
    	this.resetMeshIndices();
    	
    	return h1; // return the halfedge preceding 'e=e1' in ccw direction
    }

    public Halfedge<X> makeHole(Halfedge<X> h){
    	if(h==null) throw new Error("error making hole: h null");
    	if(h.face==null) {
    		System.out.println("makeHole: h.face null");
    		return null;
    	}
    	
    	this.facets.remove(h.getFace());

    	Halfedge<X> p=h.next;
    	h.face=null;
    	while(p!=h) {
    		p.face=null;
    		p=p.next;
    	}
    	return h;
    }

    public Halfedge<X> fillHole(Halfedge h){
    	if(h.face!=null) throw new Error("error filling hole: h not boundary edge");
    	
    	Face<X> newFace=new Face<X>();
    	this.facets.add(newFace);
    	newFace.setEdge(h);

    	Halfedge p=h.next;
    	h.face=newFace;
    	int cont=1;
    	while(p!=h) {
    		p.face=newFace;
    		p=p.next;
    	}
    	return h;
    }

    /**
     * Creates a new triangle facet within the hole incident to 'h' by connecting 
     * the tip of 'h' with two new halfedges and a new vertex. 
     * 
     * @return  the halfedge of the new edge that is incident to the new facet and the new vertex. 
     */	        
    public Halfedge<X> addTriangleToBorder(Halfedge h, X point) {
    	if(h.face!=null) 
    		throw new Error("no border edge");
    	//System.out.println("adding triangle to "+h);
    	
    	Face<X> newFace=new Face<X>();
    	Vertex<X> newVertex=new Vertex<X>(point);
    	Halfedge<X> hPrev=new Halfedge<X>();
    	Halfedge<X> hNext=new Halfedge<X>();
    	Halfedge<X> hPrevOpp=new Halfedge<X>();
    	Halfedge<X> hNextOpp=new Halfedge<X>();
    	
    	// setting the new face
    	newFace.setEdge(h);
    	// setting hPrev (halfedge preceding h in the new face)
    	hPrev.setFace(newFace);
    	hPrev.setVertex(h.getOpposite().getVertex());
    	hPrev.setPrev(hNext);
    	hPrev.setNext(h);
    	hPrev.setOpposite(hPrevOpp);
    	// setting hNext (halfedge following h in the new face)
    	hNext.setFace(newFace);
    	hNext.setVertex(newVertex);
    	hNext.setPrev(h);
    	hNext.setNext(hPrev);
    	hNext.setOpposite(hNextOpp);
    	// setting hPrevOpp (new boundary halfedge)
    	hPrevOpp.setFace(null);
    	hPrevOpp.setVertex(newVertex);
    	hPrevOpp.setPrev(h.getPrev());
    	hPrevOpp.setNext(hNextOpp);
    	hPrevOpp.setOpposite(hPrev);
    	// setting hNextOpp (the other new boundary halfedge)
    	hNextOpp.setFace(null);
    	hNextOpp.setVertex(h.getVertex());
    	hNextOpp.setPrev(hPrevOpp);
    	hNextOpp.setNext(h.getNext());
    	hNextOpp.setOpposite(hNext);
    	// updating old boundary halfedge informations
    	h.setFace(newFace);
    	h.setPrev(hPrev);
    	h.setNext(hNext);
    	// setting newVertex
    	newVertex.setEdge(hPrev); // LCA: a controler si c'est hPrev ou hNext
    	
    	// adding new facet, vertex and the four halfedges
    	this.vertices.add(newVertex);
    	this.facets.add(newFace);
    	this.halfedges.add(hPrev);
    	this.halfedges.add(hNext);
    	this.halfedges.add(hPrevOpp);
    	this.halfedges.add(hNextOpp);
    	
    	return hNext;
    }

    /**
     * A triangle with border edges is added to the polyhedral surface. <br>
     * 
     * @return  a non-border halfedge of the triangle.
     */	        
 	public Halfedge<X> makeTriangle(X p1, X p2, X p3) {    	
    	Face<X> newFace=new Face<X>();
    	Vertex<X> newVertex1=new Vertex<X>(p1);
    	Vertex<X> newVertex2=new Vertex<X>(p2);
    	Vertex<X> newVertex3=new Vertex<X>(p3);
    	Halfedge<X> e1=new Halfedge<X>();
    	Halfedge<X> e2=new Halfedge<X>();
    	Halfedge<X> e3=new Halfedge<X>();
    	Halfedge<X> e1Opp=new Halfedge<X>();
    	Halfedge<X> e2Opp=new Halfedge<X>();
    	Halfedge<X> e3Opp=new Halfedge<X>();
    	
    	// setting the new face
    	newFace.setEdge(e1);
    	// setting the new vertices (LCA: a' controler)
    	newVertex1.setEdge(e2);
    	newVertex2.setEdge(e3);
    	newVertex3.setEdge(e1);
    	// setting e1
    	e1.setFace(newFace);
    	e1.setVertex(newVertex1);
    	e1.setPrev(e3);
    	e1.setNext(e2);
    	e1.setOpposite(e1Opp);
    	// setting e2
    	e2.setFace(newFace);
    	e2.setVertex(newVertex2);
    	e2.setPrev(e1);
    	e2.setNext(e3);
    	e2.setOpposite(e2Opp);
    	// setting e3
    	e3.setFace(newFace);
    	e3.setVertex(newVertex3);
    	e3.setPrev(e2);
    	e3.setNext(e1);
    	e3.setOpposite(e3Opp);
    	// setting e1Opp (boundary halfedge opposite to e1)
    	e1Opp.setFace(null);
    	e1Opp.setVertex(newVertex3);
    	e1Opp.setPrev(e2Opp);
    	e1Opp.setNext(e3Opp);
    	e1Opp.setOpposite(e1);
    	// setting e2Opp (boundary halfedge opposite to e2)
    	e2Opp.setFace(null);
    	e2Opp.setVertex(newVertex1);
    	e2Opp.setPrev(e3Opp);
    	e2Opp.setNext(e1Opp);
    	e2Opp.setOpposite(e2);
    	// setting e3Opp (boundary halfedge opposite to e3)
    	e3Opp.setFace(null);
    	e3Opp.setVertex(newVertex2);
    	e3Opp.setPrev(e1Opp);
    	e3Opp.setNext(e2Opp);
    	e3Opp.setOpposite(e3);
    	
    	this.facets.add(newFace);
    	this.vertices.add(newVertex1);
    	this.vertices.add(newVertex2);
    	this.vertices.add(newVertex3);
    	this.halfedges.add(e1); this.halfedges.add(e1Opp);
    	this.halfedges.add(e2); this.halfedges.add(e2Opp);
    	this.halfedges.add(e3); this.halfedges.add(e3Opp);
    	
    	return e1;
    }

 	/**
 	 * Split the facet incident to <tt>h</tt> and <tt>g</tt> into two facets with a new diagonal 
 	 * between the two vertices denoted by <tt>h</tt> and <tt>g</tt> respectively. <br>
 	 * <br>
 	 * Returns <code>h->next()</code> after the operation, i.e., the new diagonal. 
 	 * The new face is to the right of the new diagonal, the old face is to the left. <br>
 	 * The time is proportional to the distance from <tt>h</tt> to <tt>g</tt> around the facet. 
 	 */	    
    public Halfedge<X> splitFacet(Halfedge<X> h, Halfedge<X> g) {
    	if(h==null || g==null) {
    		throw new Error("splitFacet: null pointer");
    	}
    	if(h.face==null || g.face==null) {
    		if(verbosity!=this.NOTHING)
    			System.out.println("splitFacet: boundary edges");
    		return null;
    	}
    	if(h.face!=g.face) {
    		if(verbosity!=this.NOTHING)
    			System.out.println("splitFacet: different incident facets");
    		return null;
    	}
    	if(h==g || h.next==g || g.next==h) {
    		if(verbosity!=this.NOTHING)
    			System.out.println("splitFace error: loops and multiple edges are not allowed");
    		return null;
    	}

    	if(h.getNext().getNext().getNext()==h) {
    		if(verbosity!=this.NOTHING)
    			System.out.println("splitFace error: triangular face");
    		return null;
    	}

		Halfedge<X> newDiagonal=new Halfedge<X>();
		Halfedge<X> oppositeNewDiagonal=new Halfedge<X>();
		Face<X> newFace=new Face<X>();

		// set face incidence relations
		newFace.setEdge(g);
		h.getFace().setEdge(h);
		newDiagonal.setFace(h.getFace());
		
		// set incidence relations between edges for the diagonals
		g.getNext().setPrev(newDiagonal);
		newDiagonal.setNext(g.getNext());
		h.getNext().setPrev(oppositeNewDiagonal);
		oppositeNewDiagonal.setNext(h.getNext());
		
		newDiagonal.setOpposite(oppositeNewDiagonal);
		oppositeNewDiagonal.setOpposite(newDiagonal);
		
		newDiagonal.setPrev(h);
		h.setNext(newDiagonal);
		
		oppositeNewDiagonal.setPrev(g);
		g.setNext(oppositeNewDiagonal);
		
		// set vertex incidence relations
		newDiagonal.setVertex(g.getVertex());
		oppositeNewDiagonal.setVertex(h.getVertex());
		
		// set face incidence relations for the new face
		g.setFace(newFace);
		Halfedge<X> pEdge=g.getNext();
		while(pEdge!=g) {
			pEdge.setFace(newFace);
			pEdge=pEdge.getNext();
		}
		
		// store the new cells (face and half-edges)
		this.halfedges.add(newDiagonal);
		this.halfedges.add(oppositeNewDiagonal);
		this.facets.add(newFace);
		
		return newDiagonal;
    } 

    /**
     * Join the the facets incident to the half-edge <tt>h</tt>.
     * <br>
     * Returns <code>h->prev()</code> after the operation, i.e., one edge of the (merged) face.
     * <br>
     * Warning: we assume that faces and edges are indexed. <br>
     * <br>
     * The time is proportional to the size of the two facets.
     * 
     * @param h  the half-edge to be removed (the two incident vertices must have degree at least three)
     */	    
    public Halfedge<X> joinFacet(Halfedge<X> h) {
    	if(h==null) {
    		throw new Error("joinFacet: half-edge not defined, null pointer");
    	}
    	if(h.face==null || h.getOpposite().getFace()==null) {
    		System.out.println("joinFacet: boundary edges");
    		return null;
    	}
    	if(h.face==h.getOpposite().getFace()) {
    		System.out.println("joinFacet: cannot be applied to the case of multiple edges and loops");
    		return null;
    	}
    	if(h==h.next) {
    		System.out.println("joinFace error: loops are not allowed");
    		return null;
    	}
    	if(h.getNext().getNext()==h) {
    		System.out.println("jointFace error: multiple edges not allowed");
    		return null;
    	}
    	if(this.vertexDegree(h.getVertex())<3) {
    		System.out.println("jointFace error: target vertex 'v' has degree less than 3");
    		return null;
    	}
    	if(this.vertexDegree(h.getOpposite().getVertex())<3) {
    		System.out.println("jointFace error: source vertex 'u' has degree less than 3");
    		return null;
    	}

    	Halfedge<X> hPrev=h.getPrev();
    	Halfedge<X> hNext=h.getNext();
    	Halfedge<X> hOppPrev=h.getOpposite().getPrev();
    	Halfedge<X> hOppNext=h.getOpposite().getNext();
    	Face<X> f1=h.getFace(); // face containing 'h'
    	Face<X> f2=h.getOpposite().getFace(); // opposite adjacent face
    	Vertex<X> u=hPrev.getVertex(); // source of 'h'
    	Vertex<X> v=h.getVertex(); // target of 'h'

    	// set incidence relations between the four halfedges
    	hPrev.setNext(hOppNext);
    	hOppNext.setPrev(hPrev);
    	
    	hNext.setPrev(hOppPrev);
    	hOppPrev.setNext(hNext);
    	
    	// set incidence relations for the vertices 'u' and 'v'
    	u.setEdge(hPrev);
    	v.setEdge(hOppPrev);

    	// set face/edge incidence relations for the merged face
    	f1.setEdge(hPrev);
    	
    	Halfedge<X> pEdge=hPrev;
    	while(pEdge!=hNext) {
    		pEdge.setFace(f1);
    		pEdge=pEdge.getNext();
    	}
    	pEdge.next.setFace(f1);

    	Halfedge<X> g=h.getOpposite();
    	// remove the two half-edges 'h' and 'g' and one face: elements are not removed, but just set to null
    	this.halfedges.set(h.index, null); // 'remove' the half-edge 'h'
    	this.halfedges.set(g.index, null); // remove the opposite half-edge
    	this.facets.set(f2.index, null); // remove face f2
    	f2.halfedge=null;
    	h.face=null;
    	h.next=null;
    	h.prev=null;
    	h.vertex=null;
    	h.opposite=null;
    	g.face=null;
    	g.next=null;
    	g.prev=null;
    	g.vertex=null;
    	g.opposite=null;

    	return hPrev;
    } 

	/**
	 * Perform the collapse of half-edge <tt>e=(u,v)</tt>.
	 * The half-edge <tt>e</tt> is shared by triangles (u,v,w) and (v,u,z).
	 */
	public void edgeCollapse(Halfedge<Point_3> e) {
		//System.out.print("Performing edge collapse...");
		if(e==null)
			return;
		
		// retrieve the cells incident to edge e
		Face<Point_3> f1=e.getFace();
		Face<Point_3> f2=e.getOpposite().getFace();
		Vertex<Point_3> u=e.getOpposite().getVertex();
		Vertex<Point_3> v=e.getVertex();
		Vertex<Point_3> w=e.getNext().getVertex();
		Vertex<Point_3> z=e.getOpposite().getNext().getVertex();
		Halfedge<Point_3> eUW=e.prev.opposite;
		Halfedge<Point_3> eWV=e.next.opposite;
		Halfedge<Point_3> eZU=e.opposite.next.opposite;
		Halfedge<Point_3> eVZ=e.opposite.prev.opposite;
		
		// update references between neighboring cells
		eUW.opposite=eWV;
		eWV.opposite=eUW;
		eZU.opposite=eVZ;
		eVZ.opposite=eZU;
		u.setEdge(eZU);
		w.setEdge(eUW);
		z.setEdge(eVZ);
		
		Halfedge<Point_3> pEdge=eWV;
		while(pEdge!=eVZ.getOpposite()) {
			pEdge.setVertex(u);
			pEdge=pEdge.getNext().getOpposite();
		}
		
		Point_3 newPoint=new Point_3(u.getPoint()); // compute new point location
		u.setPoint(newPoint);
		
		// remove old cells: 2 faces, 1 vertex, 6 halfedges
		// slow implementation: the arraylist must be updated at each removal
		this.vertices.remove(v);
		this.facets.remove(f1);
		this.facets.remove(f2);
		this.halfedges.remove(e);
		this.halfedges.remove(e.next);
		this.halfedges.remove(e.prev);
		this.halfedges.remove(e.opposite);
		this.halfedges.remove(e.opposite.next);
		this.halfedges.remove(e.opposite.prev);
		//System.out.println("done");
	}

	/**
	 * Split the half-edge <tt>h=(u,v)</tt> by inserting a new vertex, with coordinates given by the point <tt>p</tt>.
	 * As a result, a new pair of half-edges is inserted in the mesh. <br>
	 * <br>
	 * Remark: it works also for meshes with boundaries. <br>
	 * <b>Assumption</b>: the half-edge must be incident to a face: <code>h.face!=null</code> 
	 * 
	 * @param h  	the half-edge to be split
	 * @param p  	coordinates of the new vertex to insert
	 * @return 		the new inserted half-edge, having target point is <tt>p</tt>
	 */	    
	public Halfedge<X> splitEdge(Halfedge<X> h, X p) {
		if(h==null) {
			throw new Error("splitEdge: null pointer");
		}
		if(h.face==null) {
			System.out.println("Warning: trying to split a half-edge on a hole");
			return null;
		}
		
		if(this.verbosity==this.ALL) {
			System.out.print("splitting edge...");
		}
		// create the new edges, faces and vertex to be added
		Halfedge<X> hNewLeft=new Halfedge<X>();
		Halfedge<X> hNewRight=new Halfedge<X>();
		Vertex<X> newVertex=new Vertex<X>(p);

		// set the vertex incidence relations
		newVertex.setEdge(hNewLeft);
		hNewLeft.setVertex(newVertex);

		Vertex<X> u=h.getOpposite().getVertex();
		hNewRight.setVertex(u);
		u.setEdge(hNewRight);

		h.opposite.setVertex(newVertex);

		// set incidence relations for the first new halfedge
		hNewLeft.setFace(h.face);
		hNewLeft.setPrev(h.prev);
		hNewLeft.setNext(h);
		hNewLeft.setOpposite(hNewRight);
		// warning: the order in the updates does count here
		h.prev.setNext(hNewLeft); 
		h.setPrev(hNewLeft);

		// set incidence relations for the second new halfedge
		hNewRight.setFace(h.opposite.face);
		hNewRight.setPrev(h.opposite);
		hNewRight.setNext(h.opposite.next);
		hNewRight.setOpposite(hNewLeft);
		// warning: the order in the updates does count here
		h.opposite.next.setPrev(hNewRight);
		h.opposite.setNext(hNewRight);

		// add new cells to the polyhedron
		this.vertices.add(newVertex);
		this.halfedges.add(hNewLeft);
		this.halfedges.add(hNewRight);

		if(this.verbosity==this.ALL) {
			System.out.println("done");
		}
		
		return hNewLeft;
	} 

    /**
     * Reset indices for all: vertices, faces and halfedges. <br>
     * Indices correspond to the order of storage in the collections (e.g., vertex v0 is the first in the ArrayList 'vertices')
     */	
    public void resetMeshIndices() {
    	int count=0;
    	for(Vertex v: this.vertices) {
    		if(v!=null) {
    			v.index=count;
    		}
    		count++;
    	}

    	count=0;
    	for(Face f: this.facets) {
    		if(f!=null) {
    			f.index=count;
    		}
    		count++;
    	}

    	count=0;
    	for(Halfedge e: this.halfedges) {
       		if(e!=null) {
       			e.index=count;
       		}
    		count++;
    	}
    }

    /**
     * Return a string representing the list of vertices
     */
    public String verticesToString() {
        String result="List of vertices\n";
        Iterator it=this.vertices.iterator();
        int cont=0;
        while(it.hasNext()) {
        	Vertex<X> v=(Vertex<X>)it.next();
        	result=result+"v"+cont+" "+v.getPoint().toString()+"\n"; 
        	cont++;
        }  	
        return result;           
    }

    /**
     * Return a string representing the list of faces
     */
    public String facesToString() {
        String result="List of faces\n";
        Iterator it=this.facets.iterator();
        int cont=0;
        while(it.hasNext()) {
        	Face<X> f=(Face<X>)it.next();
        	result=result+"f"+cont+" ";
        	result=result+f.toString()+"\n";
        	cont++;
        }  	
        return result;           
    }

    /**
     * Return a string representing the list of faces
     */
    public String edgesToString() {
        String result="List of edges\n";
        Iterator it=this.halfedges.iterator();
        int cont=0;
        while(it.hasNext()) {
        	Halfedge<X> e=(Halfedge<X>)it.next();
        	result=result+"e"+cont+" ";
        	//result=result+vertices.indexOf(e.getVertex())+"\n";
        	result=result+e.toString()+"\n";
        	cont++;
        }  	
        return result;           
    }
    
}