package Jcg.polyhedron;

import Jcg.geometry.*;

/**
 * A class representing a face of a polyhedral (surface or planar) mesh (<em>Half-edge</em> data structure)
 */
public class Face<X extends Point_> {

	/** a reference to a half-edge incident to this face */
	public Halfedge<X> halfedge=null;
	
	/** an integer that can be used to index faces (useful for some applications) */
	public int index;
	
	/** an integer that can be used to tag faces (useful for dealing with colors, ...) */
	public int tag;

    public Face() {}
        
	/**
	 * Compute the degree of the face (number of bounding edges)
	 */	
    public int degree() {
    	Halfedge<X> e,p;
    	if(this.halfedge==null) return 0;
    	
    	e=halfedge; p=halfedge.next;
    	int cont=1;
    	while(p!=e) {
    		cont++;
    		p=p.next;
    	}
    	return cont;
    }
    
	/**
	 * true iff the face is a triangle
	 */	
	public boolean isTriangle() {
		Halfedge<X> h=this.getEdge();
		if(h.getNext().getNext().getNext()!=h)
			return false;
		return true;
	}

	/**
	 * true iff the face is a quad
	 */	
	public boolean isQuad() {
		if(this.degree()==4)
			return true;
		return false;
	}

    
	/**
	 * Return an array storing the indices of the vertices in the face<p>
	 * 
	 * @param polyhedron 	the input mesh
	 * @return 	array storing indices of the vertices of the face
	 */
    public int[] getVertexIndices(Polyhedron_3<X> polyhedron) {
    	int d=this.degree();
    	int[] result=new int[d];
    	Vertex<X> v;
    	
    	Halfedge<X> e,p;
    	if(this.halfedge==null) return null;
    	
    	e=halfedge; p=halfedge.next;
    	v=e.getVertex();
    	result[0]=polyhedron.vertices.indexOf(v);
    	int cont=1;
    	while(p!=e) {
    		v=p.getVertex();
    		result[cont]=polyhedron.vertices.indexOf(v);
    		cont++;
    		p=p.next;
    	}
    	return result;
    }

	/**
	 * Return an array storing the geometric coordinates of the vertices of the face<p>
	 * 
	 * @return the points (2d or 3d) corresponding to the geometric coordinates of the face
	 */
    public Point_[] getVertexCoordinates() {
    	int d=this.degree();
    	Point_[] result=new Point_[d];
    	Vertex<X> v;
    	
    	Halfedge<X> e,p;
    	if(this.halfedge==null) return null;
    	
    	e=halfedge; p=halfedge.next;
    	v=e.getVertex();
    	result[0]=v.getPoint();
    	int cont=1;
    	while(p!=e) {
    		v=p.getVertex();
    		result[cont]=v.getPoint();
    		cont++;
    		p=p.next;
    	}
    	return result;
    }

	/**
	 * Compute and return the barycenter (2d or 3d) of the vertices of the face<p>
	 * 
	 * @return the barycenter of the vertices of the face
	 */
    public Point_ getBarycenter() {
    	Point_[] points=this.getVertexCoordinates();
		Point_ bar=null; // barycenter to compute
		if(points[0] instanceof Point_2)
			bar=new Point_2();
		else if(points[0] instanceof Point_3)
			bar=new Point_3();
		else
			throw new Error("Error: wrong type for points ");
		
		bar.barycenter(points);
		return bar;
    }

    
    public void setEdge(Halfedge<X> halfedge) { this.halfedge=halfedge; }
    public Halfedge<X> getEdge() { return this.halfedge; }
        
    public String toString(){
    	String result="";
    	Halfedge<X> e,p;
    	if(this.halfedge==null) {
    		System.out.println("face.toString() error: null incident halfedge");
    		return null;
    	}
    	
    	e=halfedge;
    	p=halfedge.next;
    	result=result+e.getVertex().toString();
    	while(p!=e) {
    		result=result+"\t"+p.getVertex().toString();
    		p=p.next;
    	}
    	return result;
    }

}
