package Jcg.meshgeneration;
import Jcg.geometry.*;
import Jcg.mesh.MeshBuilder;
import Jcg.mesh.SharedVertexRepresentation;
import Jcg.polyhedron.*;

import java.awt.Color;
import java.util.*;

/**
 * Generate a spherical mesh, performing the recursive subdivision of an icosahedron
 * 
 * @author Luca Castelli Aleardi (Ecole Polytechnique, 2019)
 *
 */
public class SphereGeneration {
	/** the generated mesh to output */
	public Polyhedron_3<Point_3> polyhedron3D;
	
	private HashSet<Vertex<Point_3>> originalVertices; // store the references of the original vertices
	
	public SphereGeneration() {
		int n=12; // create an icosahedron as starting spherical polyhedron
		int f=20;
        Point_3[] points=new Point_3[n];
        int[] faceDegrees=new int[f];
        int[][] faces=new int[f][3];
        
        points[0]=new Point_3(0.0, 0.0, 2.0);
        points[1]=new Point_3(1.788854, 0.000000, 0.894427);
        points[2]=new Point_3(0.552786, 1.701302, 0.894427);
        points[3]=new Point_3(-1.447214, 1.051462, 0.894427);
        points[4]=new Point_3(-1.447214, -1.051462, 0.894427);
        points[5]=new Point_3(0.552786, -1.701302, 0.894427);
        points[6]=new Point_3(1.447214, 1.051462, -0.894427);
        points[7]=new Point_3(-0.552786, 1.701302, -0.894427);
        points[8]=new Point_3(-1.788854, 0.000000, -0.894427);
        points[9]=new Point_3(-0.552786, -1.701302, -0.894427);
        points[10]=new Point_3(1.447214, -1.051462, -0.894427);
        points[11]=new Point_3(0.0, 0.0, -2.0);
        
        // normalized all points
        Point_3 origin=new Point_3();
        for(int i=0;i<n;i++) {
        	points[i].multiply(1./points[i].distanceFrom(origin).doubleValue());
        }
        
        for(int i=0;i<f;i++)
        	faceDegrees[i]=3;
        
        faces[0]=new int[] {2, 0, 1};
        faces[1]=new int[] {3, 0, 2};
        faces[2]=new int[] {4, 0, 3};
        faces[3]=new int[] {5, 0, 4};
        faces[4]=new int[] {1, 0, 5};
        faces[5]=new int[] {2, 1, 6};
        faces[6]=new int[] {7, 2, 6};
        faces[7]=new int[] {3, 2, 7};
        faces[8]=new int[] {8, 3, 7};
        faces[9]=new int[] {4, 3, 8};
        faces[10]=new int[] {9, 4, 8};
        faces[11]=new int[] {5, 4, 9};
        faces[12]=new int[] {10, 5, 9};
        faces[13]=new int[] {6, 1, 10};
        faces[14]=new int[] {1, 5, 10};
        faces[15]=new int[] {6, 11, 7};
        faces[16]=new int[] {7, 11, 8};
        faces[17]=new int[] {8, 11, 9};
        faces[18]=new int[] {9, 11, 10};
        faces[19]=new int[] {10, 11, 6};
		
		MeshBuilder<Point_3> builder=new MeshBuilder<Point_3>();
		this.polyhedron3D=builder.createMesh(points, faceDegrees, faces);
		//this.polyhedron3D.isValid(false);
	}
	
	/**
	 * The main method performing the subdivision process
	 */
	public void run(int iterations) {
		int f=this.polyhedron3D.sizeOfFacets(); // faces of the initial icosahedron
		int F=(int)Math.pow(4, iterations)*f; // size of the final mesh
		System.out.print("Generating a spherical triangle mesh ("+F+" faces)...");
		for(int i=0;i<iterations;i++)
			this.runOnce();
		System.out.println("done");
		
		this.polyhedron3D.isValid(false);
	}

	/**
	 * The main method performing the subdivision process
	 */
	private void runOnce() {
		
		// useful for distinguishing between original vertices and the new ones (added by the subdivision scheme)
		originalVertices=new HashSet<Vertex<Point_3>>();
		for(Vertex<Point_3> v: this.polyhedron3D.vertices) {
			this.originalVertices.add(v);
		}
		
		HashMap<Halfedge<Point_3>, Point_3> edgePoints=this.computeEdgePoints();
		splitEdges(edgePoints); // perform the split of all (original) edges
		
		// remark: cannot access the Arraylist polyhedron.facets, because of concurrency
		LinkedList<Face<Point_3>> facesToSubdivide=new LinkedList<Face<Point_3>>();
		for(Face<Point_3> f: this.polyhedron3D.facets) {
			facesToSubdivide.add(f);
		}
		for(Face<Point_3> f: facesToSubdivide)
			subdivideFace(f);
		
	}

	/**
	 * Splits all edges by inserting a new vertex
	 */
	private void splitEdges(HashMap<Halfedge<Point_3>, Point_3> edgePoints) {
		//System.out.print("Splitting edges...");
		// set of half-edges which have been already split
		LinkedList<Halfedge<Point_3>> edgesToSplit=new LinkedList<Halfedge<Point_3>>();
		for(Halfedge<Point_3> h: this.polyhedron3D.halfedges) {
			h.tag=1; // set the edge to be split;
			edgesToSplit.add(h);
		}
		
		while(edgesToSplit.size()>0) {
			Halfedge<Point_3> h=edgesToSplit.pop();
			if(h.tag==1) { // the halfedge, and its opposite, must be split
				Point_3 midPoint=edgePoints.get(h);
				this.polyhedron3D.splitEdge(h, midPoint);
				
				h.tag=0;
				h.opposite.tag=0;
			}
		}
	}
	
	/**
	 * Perform the subdivision of a face into 4 triangular sub-faces
	 * Edges must already be split: the face has degree 3+3
	 */
	private void subdivideFace(Face<Point_3> f) {
		if(f==null || f.degree()<4)
			return;
		Halfedge<Point_3> h1=f.getEdge();
		if(originalVertices.contains(h1.getVertex())==true) // the half-edge must point to a "new midpoint" vertex
			h1=h1.getNext();
		
		Halfedge<Point_3> g1=h1.getNext().getNext();
		Halfedge<Point_3> h2=this.polyhedron3D.splitFacet(h1, g1);
		Halfedge<Point_3> g2=h2.getNext().getNext();
		Halfedge<Point_3> g3=this.polyhedron3D.splitFacet(h2, g2);
		this.polyhedron3D.splitFacet(h1, g3);
	}

	/**
	 * Compute the midpoint of the given the half-edge 'h'
	 */
	private Point_3 computeEdgePoint(Halfedge<Point_3> h) {
		Point_3 origin=new Point_3();
		// retrieve the coordinates of the four points incident to the two faces sharing h
		Point_3 v=h.getVertex().getPoint();
		Point_3 u=h.getOpposite().getVertex().getPoint();
		
		Point_3[] points={u, v};
		Double[] coefficients={1./2, 1./2};
		Point_3 result=Point_3.linearCombination(points, coefficients);
		
		// normalize
		result.multiply(1./result.distanceFrom(origin).doubleValue());
		
		return result;
	}

	/**
	 * Compute all new edge points and store the result in an HashMap
	 */
	private HashMap<Halfedge<Point_3>, Point_3> computeEdgePoints() {
		// store the new edge points in an hash table
		HashMap<Halfedge<Point_3>, Point_3> result=new HashMap<Halfedge<Point_3>, Point_3>();
		for(Halfedge<Point_3> h: this.polyhedron3D.halfedges) {
			//if(result.containsKey(h.getOpposite())==false) // only for reducing the number of entries in the table
				result.put(h, computeEdgePoint(h));
		}
		
		return result;
	}
	
	public static double approx(double x, int prec) {
		double p=(int)Math.pow(10, prec);
		return ((int)(x*p))/p;
	}

}
