package Jcg.meshgeneration;
import Jcg.geometry.Point_;
import Jcg.geometry.Point_3;
import Jcg.mesh.IO;
import Jcg.mesh.MeshLoader;
import Jcg.polyhedron.Face;
import Jcg.polyhedron.Halfedge;
import Jcg.polyhedron.Polyhedron_3;
import Jcg.polyhedron.Vertex;

public class RegularGridGeneration {

    /**
     * Output a regular grid
     * 
     * @param mesh a triangulation (Half-edge representation)
     * 
     */
    public static void writeRegularGrid(int n, int m, String filename){
    	System.out.print("Creating a regular grid of size "+n+"x"+m+"...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int nVertices=(n+1)*(m+1);
    	int nFaces=2*n*m;
    	
    	// file header
    	Jcg.io.IO.println("OFF");
    	Jcg.io.IO.println(nVertices+" "+nFaces+" "+0);
    	    	
    	// writing vertex coordinates
    	double deltaX=1./n;
    	double deltaY=1./m;
    	for(int i=0;i<n+1;i++) { // iterate over all vertices
    		for(int j=0;j<m+1;j++) {
    			double x, y, z;
    			x=i*deltaX;
    			y=j*deltaY;
    			z=0.;

    			Jcg.io.IO.println(""+x+" "+y+" "+z);
    		}
    	}
    	
      	// write face/vertex incidence relations
    	for(int i=0;i<n;i++){ // iterate over all faces
    		for(int j=0;j<m;j++) {
    			int index0=j*(n+1)+i; // (i, j)
    			int index1=j*(n+1)+i+1; // (i+1, j)
    			int index2=(j+1)*(n+1)+(i+1); // (i+1, j+1)
    			int index3=(j+1)*(n+1)+i; // (i, j+1)
    			Jcg.io.IO.println(3+" "+index0+" "+index1+" "+index2);
    			Jcg.io.IO.println(3+" "+index2+" "+index3+" "+index0);
    		}
    	}
    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

    /**
     * Output a triangular grid
     * 
     * @param mesh a triangulation (Half-edge representation)
     * 
     */
    public static void writeTriangularGrid(int n, int m, String filename){
    	System.out.print("Creating a regular grid of size "+n+"x"+m+"...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int nVertices=(n+1)*(m+1)+3;
    	int nFaces=2*n*m+(2*n+2*m)+4;
    	
    	// file header
    	Jcg.io.IO.println("OFF");
    	Jcg.io.IO.println(nVertices+" "+nFaces+" "+0);
    	    	
    	// writing vertex coordinates
    	double deltaX=1./n;
    	double deltaY=1./m;
    	for(int i=0;i<n+1;i++) { // iterate over all vertices
    		for(int j=0;j<m+1;j++) {
    			double x, y, z;
    			x=i*deltaX;
    			y=j*deltaY;
    			z=0.;

    			Jcg.io.IO.println(""+x+" "+y+" "+z);
    		}
    	}
    	// coordinates of vertices on the outer face
    	Jcg.io.IO.println(""+(-2.0)+" "+(-2.0)+" "+0.0); // v0
    	Jcg.io.IO.println(""+(3.0)+" "+(-2.0)+" "+0.0); // v1
    	Jcg.io.IO.println(""+(0.5)+" "+(2.5)+" "+0.0); // v2
    	
    	// index of vertices on the outer face
    	int v0=nVertices-3;
    	int v1=nVertices-2;
    	int v2=nVertices-1;
    	
      	// write face/vertex incidence relations
    	Jcg.io.IO.println(3+" "+v0+" "+v2+" "+v1); // outer face (v0, v2, v1), having opposite orientation
    	// inner faces incident to v0
    	int indexA=0, indexB=0;
    	for(int j=0;j<m;j++) {
    		indexA=j; // (0, j)
    		indexB=(j+1); // (0, j+1)
    		Jcg.io.IO.println(3+" "+v0+" "+indexA+" "+indexB);
    	}
    	Jcg.io.IO.println(3+" "+v0+" "+indexB+" "+v2);
    	for(int i=0;i<n;i++){ 
    		indexA=(m+1)*i; // (i, j)
    		indexB=(m+1)*(i+1); // (i+1, j)
    		Jcg.io.IO.println(3+" "+indexB+" "+indexA+" "+v0);
    	}
    	Jcg.io.IO.println(3+" "+v1+" "+indexB+" "+v0);
    	
    	// inner faces incident to v1
    	for(int j=0;j<m;j++) {
    		indexA=(m+1)*n+j; // (n, j)
    		indexB=(m+1)*n+(j+1); // (n, j+1)
    		Jcg.io.IO.println(3+" "+v1+" "+indexB+" "+indexA);
    	}
    	Jcg.io.IO.println(3+" "+v1+" "+v2+" "+indexB);

    	// inner faces incident to v2
    	for(int i=0;i<n;i++){ 
    		indexA=(n+1)*i+m; // (i, m)
    		indexB=(n+1)*(i+1)+m; // (i+1, j)
    		Jcg.io.IO.println(3+" "+indexA+" "+indexB+" "+v2);
    	}

    	// inner faces, not incident to (v0, v1, v2)
    	for(int i=0;i<n;i++){ // iterate over all faces
    		for(int j=0;j<m;j++) {
    			int index0=j*(n+1)+i; // (i, j)
    			int index1=j*(n+1)+i+1; // (i+1, j)
    			int index2=(j+1)*(n+1)+(i+1); // (i+1, j+1)
    			int index3=(j+1)*(n+1)+i; // (i, j+1)
    			Jcg.io.IO.println(3+" "+index2+" "+index1+" "+index0);
    			Jcg.io.IO.println(3+" "+index0+" "+index3+" "+index2);
    		}
    	}
    	
    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

    /**
     * Output a regular cylindric grid
     * 
     * @param mesh a triangulation (Half-edge representation)
     * 
     */
    public static void writeRegularCylinder(int n, int m, String filename){
    	System.out.print("Creating a "+n+"x"+m+" cylinder...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int nVertices=(n)*(m+1)+2;
    	int nFaces=2*n*m+(2*n);
    	
    	// file header
    	Jcg.io.IO.println("OFF");
    	Jcg.io.IO.println(nVertices+" "+nFaces+" "+0);
    	    	
    	// writing vertex coordinates
    	double deltaAlpha=2*Math.PI/n;
    	double height=50.;
    	double deltaZ=height/m;
    	for(int j=0;j<m+1;j++) { // iterate over all vertices
    		for(int i=0;i<n;i++) {
    			double x, y, z;
    			x=Math.cos(i*deltaAlpha);
    			y=Math.sin(i*deltaAlpha);
    			z=-height/2.0+j*deltaZ;

    			Jcg.io.IO.println(""+x+" "+y+" "+z);
    		}
    	}
    	Jcg.io.IO.println(""+0.0+" "+0.0+" "+(-height/2.)); // south pole
    	Jcg.io.IO.println(""+0.0+" "+0.0+" "+height/2.); // north pole
    	
      	// write face/vertex incidence relations
		int indexSouth=nVertices-2;
		Jcg.io.IO.println(3+" "+0+" "+(n-1)+" "+indexSouth);
		for(int i=n-1;i>0;i--) {
			int index0=i;
			int index1=i-1;
			
			Jcg.io.IO.println(3+" "+index0+" "+index1+" "+indexSouth);
		}

		int indexNorth=nVertices-1;
		Jcg.io.IO.println(3+" "+(nVertices-3)+" "+(nVertices-2-n)+" "+indexNorth);
		for(int i=0;i<n-1;i++) {
			int index0=nVertices-2-i-1;
			int index1=nVertices-2-i-2;
			
			Jcg.io.IO.println(3+" "+index1+" "+index0+" "+indexNorth);
		}

     	for(int i=0;i<n-1;i++){ // iterate over all faces
    		for(int j=0;j<m;j++) {
    			int index0=j*(n)+i; // (i, j)
    			int index1=j*(n)+i+1; // (i+1, j)
    			int index2=(j+1)*(n)+(i+1); // (i+1, j+1)
    			int index3=(j+1)*(n)+i; // (i, j+1)
    			
    			Jcg.io.IO.println(3+" "+index0+" "+index1+" "+index2);
    			Jcg.io.IO.println(3+" "+index2+" "+index3+" "+index0);
    		}
    	}

		for(int j=0;j<m;j++) {
			int index0=(j+1)*n-1;
			int index1=j*n;
			int index2=(j+1)*n;
			int index3=(j+2)*n-1; // (i, j+1)
			
			Jcg.io.IO.println(3+" "+index0+" "+index1+" "+index2);
			Jcg.io.IO.println(3+" "+index2+" "+index3+" "+index0);
		}
		
   	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

    /**
     * Output a regular globe
     * 
     * @param mesh a triangulation (Half-edge representation)
     * 
     */
    public static void writeRegularGlobe(int n, String filename){
    	System.out.print("Creating a "+n+"x"+n+" globe...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int m=n-1;
    	int nVertices=(n)*(m+1)+2;
    	int nFaces=2*n*m+(2*n);
    	
    	// file header
    	Jcg.io.IO.println("OFF");
    	Jcg.io.IO.println(nVertices+" "+nFaces+" "+0);
    	    	
    	// writing vertex coordinates
    	double deltaAlpha=2*Math.PI/n;
    	double deltaTheta=Math.PI/(n+1);
    	double radius=1.0;
    	Point_3 origin=new Point_3(0., 0., 0.);
    	for(int j=0;j<m+1;j++) { // iterate over all vertices
    		for(int i=0;i<n;i++) {
    			double x, y, z;
    			double phi=i*deltaAlpha;
    			double theta=(j+1)*deltaTheta;
    			x=radius*Math.sin(theta)*Math.cos(phi);
    			y=radius*Math.sin(theta)*Math.sin(phi);
    			z=-radius*Math.cos(theta);

    			Jcg.io.IO.println(""+x+" "+y+" "+z);
    		}
    	}
    	Jcg.io.IO.println(""+0.0+" "+0.0+" "+"-1.0"); // south pole
    	Jcg.io.IO.println(""+0.0+" "+0.0+" "+"1.0"); // north pole
    	
      	// write face/vertex incidence relations
 		int indexSouth=nVertices-2;
		Jcg.io.IO.println(3+" "+0+" "+(n-1)+" "+indexSouth);
		for(int i=n-1;i>0;i--) {
			int index0=i;
			int index1=i-1;
			
			Jcg.io.IO.println(3+" "+index0+" "+index1+" "+indexSouth);
			//Jcg.io.IO.println(3+" "+indexSouth+" "+index1+" "+index0);
		}

		int indexNorth=nVertices-1;
		Jcg.io.IO.println(3+" "+(nVertices-3)+" "+(nVertices-2-n)+" "+indexNorth);
		for(int i=0;i<n-1;i++) {
			int index0=nVertices-2-i-1;
			int index1=nVertices-2-i-2;
			
			Jcg.io.IO.println(3+" "+index1+" "+index0+" "+indexNorth);
			//Jcg.io.IO.println(3+" "+indexNorth+" "+index0+" "+index1);
		}

   	for(int i=0;i<n-1;i++){ // iterate over all faces
    		for(int j=0;j<m;j++) {
    			int index0=j*(n)+i; // (i, j)
    			int index1=j*(n)+i+1; // (i+1, j)
    			int index2=(j+1)*(n)+(i+1); // (i+1, j+1)
    			int index3=(j+1)*(n)+i; // (i, j+1)
    			
    			Jcg.io.IO.println(3+" "+index0+" "+index1+" "+index2);
    			Jcg.io.IO.println(3+" "+index2+" "+index3+" "+index0);
    			//Jcg.io.IO.println(3+" "+index2+" "+index1+" "+index0);
    			//Jcg.io.IO.println(3+" "+index0+" "+index3+" "+index2);
    		}
    	}

		for(int j=0;j<m;j++) {
			int index0=(j+1)*n-1;
			int index1=j*n;
			int index2=(j+1)*n;
			int index3=(j+2)*n-1; // (i, j+1)
			
			Jcg.io.IO.println(3+" "+index0+" "+index1+" "+index2);
			Jcg.io.IO.println(3+" "+index2+" "+index3+" "+index0);
			//Jcg.io.IO.println(3+" "+index2+" "+index1+" "+index0);
			//Jcg.io.IO.println(3+" "+index0+" "+index3+" "+index2);
		}
		
    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

	public static Polyhedron_3<Point_3> star(String filename, int rootFace, int k) {
		Polyhedron_3<Point_3> mesh=MeshLoader.getSurfaceMesh(filename); // store the input mesh and graph

		Face root=mesh.facets.get(rootFace);
		
		System.out.println("Creating a stacked triangulation");
		int nF=mesh.sizeOfFacets();
		for(int j=0;j<k;j++) {
			for(int i=0;i<nF;i++) {
				Face f=mesh.facets.get(i);
				if(f!=root) {
					System.out.println("Subdividing face "+i);
					mesh.createCenterVertex(f, new Point_3(0., 0., 0.));
				}
			}
			//n=mesh.sizeOfVertices();
		}
		mesh.isValid(true);
		System.out.println("stacked triangulation created");
		return mesh;
	}

	public static void save(Polyhedron_3<Point_3> mesh, String filename) {
		IO.writePolyedronToOFF(mesh, filename);
	}

}
