package Jcg.mesh;

import java.awt.Color;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.HashMap;

import Jcg.geometry.*;
import Jcg.polyhedron.*;
import Jcg.triangulations2D.TriangulationDS_2;
import tc.TC;
import Jcg.triangulations2D.TriangulationDSVertex_2;
import Jcg.triangulations2D.TriangulationDSFace_2;

/**
 * A vertex shared representation of a mesh.
 * For each face we store the indices of the incident vertices
 *
 * @author Luca Castelli Aleardi (INF562-INF555, 2012-2025)
 *
 */
public class SharedVertexRepresentation {
	    
	public int sizeVertices;
	public int sizeFaces;
	public int sizeHalfedges;
	
    public int[][] faces;
    public int[] faceDegrees;
    public Point_3[] points;

    public Color[] faceColors=null;
    
    /**
     * Create a shared vertex representation
     * 
     * @param n  number of vertices
     * @param f  number of faces
     * @param he  number of half-edges
     * @param points  vertex coordinates (3D)
     */
    public SharedVertexRepresentation(int n, int f, int he, int[][] faces, int[] faceDegrees, Point_3[] points) {
    	System.out.print("Creating a shared vertex representation...");
    	
    	this.sizeVertices=n;
    	this.sizeFaces=f;
    	this.sizeHalfedges=he;
    	this.faces=faces;
    	this.faceDegrees=faceDegrees;
    	this.points=points;
    	
    	System.out.println("done");
    }

    /**
     * Create a shared vertex representation from a collection of 3D triangles
     */
    public SharedVertexRepresentation(Collection<Triangle_3> faces) {
    	System.out.print("Creating a shared vertex representation (from collection of triangles)...");
    	this.faces = new int[faces.size()][];
    	faceDegrees = new int[faces.size()];
    	faceColors = new Color[faces.size()];
    	LinkedHashMap<Point_3,Integer> vert = new LinkedHashMap<Point_3,Integer> ();

    	int ind=0;
    	int indF = 0;
    	for (Triangle_3 f : faces) {
    		for (int i=0; i<3; i++)
    			if (!vert.containsKey(f.vertex(i)))
    				vert.put(f.vertex(i), ind++);
    		faceDegrees[indF] = 3;
    		faceColors[indF] = Color.gray;
    		this.faces[indF++] = new int[]{vert.get(f.vertex(0)), vert.get(f.vertex(1)), vert.get(f.vertex(2))};
    	}    	
    	points = vert.keySet().toArray(new Point_3[0]);
    	System.out.println("done");
    }
    
    /**
     * Create a shared vertex representation from an OBJ file (using TC library)
     */
    public static SharedVertexRepresentation readOBJ(String filename){
    	System.out.print("Reading a collection of polygonal faces from OBJ format (shared vertex representation): ");
    	System.out.println(filename);
    	boolean hasFaceColors=false;
    	long startTime=System.nanoTime(), endTime; // for evaluating time performances
    	
    	double x, y, z;
    	Jcg.io.IO.readTextFile(filename);
    	String line, w[];
            
    	int sizeVertices=0, sizeFaces=0;            
    	System.out.print("\tCounting vertices and faces...");

    	while(TC.finEntree()==false) {
    		line=Jcg.io.IO.readLine();
    		w = Jcg.io.IO.wordsFromString(line);
    		if(w!=null && w.length>0 && w[0].charAt(0)!='#') { // skip empty lines an comments
    			if(w[0].equals("v"))
    				sizeVertices++;
    			if(w[0].equals("f"))
    				sizeFaces++;
    		}
    	}
    	System.out.println("done ("+sizeVertices+" vertices, "+sizeFaces+" faces)");

    	Point_3[] points=new Point_3[sizeVertices];
    	int[] faceDegrees=new int[sizeFaces];
    	int[][] faces=new int[sizeFaces][];
    	Color[] faceColors=new Color[sizeFaces];

    	int i=0;
    	Point_3 point;
    	System.out.print("\tReading vertex coordinates and faces...");

    	// start second pass
    	int countVertices=0, countFaces=0, countHalfedges=0;
    	Jcg.io.IO.readTextFile(filename);
    	while(TC.finEntree()==false) {
    		line=Jcg.io.IO.readLine();
    		w = Jcg.io.IO.wordsFromString(line);
    		if(w!=null && w.length>0 && w[0].charAt(0)!='#') { // skip empty lines an comments
    			if(w[0].equals("v")) {
    				x=(new Double(w[1])).doubleValue();
    				y=(new Double(w[2])).doubleValue();
    				z=(new Double(w[3])).doubleValue();

    				points[countVertices]=new Point_3(x,y,z);
    				countVertices++;
    			}
    			else if(w[0].equals("f")) {
               		int degree=w.length-1;
            		faceDegrees[countFaces]=degree;
            		//System.out.print("face"+countFaces+": "+degree);
            		faces[countFaces]=new int[faceDegrees[countFaces]];
            		
            		for(int j=0; j<faceDegrees[countFaces]; j++) {
            			int index=Integer.parseInt(w[j+1])-1; // vertex indices must be shifted by -1
            			faces[countFaces][j]=index;
            			//System.out.print("\t"+faces[countFaces][j]);
            			countHalfedges++;
                    }
            		//System.out.println();
            		countFaces++;
    			}
    		}
    	}
    	System.out.println("done ("+countVertices+" vertices, "+countFaces+" faces, "+countHalfedges+" halfedges)");
    	Jcg.io.IO.readStandardInput();

    	endTime=System.nanoTime();
    	double duration=(double)(endTime-startTime)/1000000000.;
    	System.out.print("Triangle soup loaded from from OBJ file");
    	System.out.println(" ("+duration+" seconds)");

    	return new SharedVertexRepresentation(countVertices, countFaces, countHalfedges, faces, faceDegrees, points);
    }
    
    /**
     * Create a mesh representation from an OFF file (using TC library)
     */
    public SharedVertexRepresentation(String filename){
    	System.out.print("Reading a collection of polygonal faces from OFF format (shared vertex representation): ");
    	System.out.println(filename);
    	boolean hasFaceColors=false;
    	long startTime=System.nanoTime(), endTime; // for evaluating time performances
    	
    	double x, y, z;
    	Jcg.io.IO.readTextFile(filename);
    	String line;
            
            line=Jcg.io.IO.readLine(); // first line is empty
            line=Jcg.io.IO.readLine();
            String[] w=Jcg.io.IO.wordsFromString(line);
            sizeVertices=Integer.parseInt(w[0]);
            sizeFaces=Integer.parseInt(w[1]);            
            
            this.points=new Point_3[sizeVertices];
            this.faceDegrees=new int[sizeFaces];
            this.faces=new int[sizeFaces][];
            this.faceColors=new Color[sizeFaces];
            
            int i=0;
            Point_3 point;
            System.out.print("\tReading vertices...");
            
            while(i<sizeVertices) {
                line=Jcg.io.IO.readLine();
                w = Jcg.io.IO.wordsFromString(line);
                if(w!=null && w.length>0 && w[0].charAt(0)!='#') { // skip empty lines an comments
                	x=(new Double(w[0])).doubleValue();
                	y=(new Double(w[1])).doubleValue();
                	z=(new Double(w[2])).doubleValue();
                
                	point=new Point_3(x,y,z);
                	points[i]=point;
                	i++;
                }
            }
            System.out.println("done "+sizeVertices+" vertices");
            
            System.out.print("\tReading face degrees...");
            i=0;
            while(i<sizeFaces){
            	line = Jcg.io.IO.readLine();
            	if(line == null)  {
            		throw new Error("error: end of file reached before reading all faces");
            	}
            	String[] words=Jcg.io.IO.wordsFromString(line);
            	if(words!=null && words.length>1) {
            		int degree=Integer.parseInt(words[0]);
            		faceDegrees[i]=degree; // first element encodes the degree
            		faces[i]=new int[faceDegrees[i]];
            		
            		for(int j=0; j<faceDegrees[i]; j++) {
            			faces[i][j]=Integer.parseInt(words[j+1]);
            			sizeHalfedges++;
                    }
            		
            		//reading face colors
            		if(words.length>faceDegrees[i]+1) {
            			if(words.length<(1+degree+3))
            				throw new Error("Error: missing color component for face f"+i);
            			int r=Integer.parseInt(words[degree+1]); // red component of the color
            			int g=Integer.parseInt(words[degree+2]); // green component of the color
            			int b=Integer.parseInt(words[degree+3]); // blue component of the color
            			this.faceColors[i]=new Color(r, g, b);
            			hasFaceColors=true;
            		}
            		
            		i++;
                }         
            }
            System.out.println("done "+sizeFaces+" faces");
            Jcg.io.IO.readStandardInput();
            
            endTime=System.nanoTime();
            double duration=(double)(endTime-startTime)/1000000000.;
            System.out.print("Triangle soup loaded from from OFF file");
            if(hasFaceColors==true)
            	System.out.print(" [with face colors] ");
            System.out.println(" ("+duration+" seconds)");
    }

    /**
     * Create a minimal triangle mesh representation from an OFF file (using TC library): no colors, no vertex coordinates
     */
    public SharedVertexRepresentation(String filename, boolean minimal){
    	System.out.print("Reading a collection of polygonal faces from OFF format (shared vertex representation): ");
    	System.out.println(filename);
    	boolean hasFaceColors=false;
    	
        Runtime runtime = Runtime.getRuntime();
        runtime.gc();
        long usedMemory1 = runtime.totalMemory() - runtime.freeMemory();
    	long startTime=System.nanoTime(), endTime; // for evaluating time performances
    	
    	double x, y, z;
    	Jcg.io.IO.readTextFile(filename);
    	String line;
            
    	line=Jcg.io.IO.readLine(); // first line is empty
    	line=Jcg.io.IO.readLine();
    	String[] w=Jcg.io.IO.wordsFromString(line);
    	sizeVertices=Integer.parseInt(w[0]);
    	sizeFaces=Integer.parseInt(w[1]);            

    	this.faceDegrees=new int[sizeFaces];
    	this.faces=new int[sizeFaces][];

    	int i=0;
    	Point_3 point;
    	System.out.print("\tReading vertices...");

    	while(i<sizeVertices) {
    		line=Jcg.io.IO.readLine();
    		w = Jcg.io.IO.wordsFromString(line);
    		if(w!=null && w.length>0 && w[0].charAt(0)!='#') { // skip empty lines an comments
    			i++;
    		}
    	}
    	System.out.println("done "+sizeVertices+" vertices");

    	System.out.print("\tReading face degrees...");
    	i=0;
    	while(i<sizeFaces){
    		line = Jcg.io.IO.readLine();
    		if(line == null)  {
    			throw new Error("error: end of file reached before reading all faces");
    		}
    		String[] words=Jcg.io.IO.wordsFromString(line);
    		if(words!=null && words.length>1) {
    			int degree=Integer.parseInt(words[0]);
    			faceDegrees[i]=degree; // first element encodes the degree
    			faces[i]=new int[faceDegrees[i]];

    			for(int j=0; j<faceDegrees[i]; j++) {
    				faces[i][j]=Integer.parseInt(words[j+1]);
    				sizeHalfedges++;
    			}

    			i++;
    		}         
    	}
    	System.out.println("done "+sizeFaces+" faces");
    	Jcg.io.IO.readStandardInput();

		long usedMemory2 = runtime.totalMemory() - runtime.freeMemory();
		long memory=(usedMemory2-usedMemory1); // Bytes
		long memoryMB=(usedMemory2-usedMemory1)/(1024L*1024L); // MBytes
    	endTime=System.nanoTime();
    	double duration=(double)(endTime-startTime)/1000000000.;
    	System.out.print("Triangle soup loaded from from OFF file");
    	if(hasFaceColors==true)
    		System.out.print(" [with face colors] ");
    	System.out.println(" ("+duration+" seconds, "+memoryMB+" MB of memory used)");
    }

    /**
     * Create a mesh representation from a polyhedron (half-edge data structure)
     */
	public SharedVertexRepresentation(TriangulationDS_2<Point_3> mesh){
    	System.out.print("Creating Shared Vertex Representation from TriangulationDS_2... ");
    	this.sizeVertices=mesh.sizeOfVertices();
    	this.sizeFaces=mesh.sizeOfFaces();           
    	
    	this.points=new Point_3[sizeVertices];
    	this.faceDegrees=new int[sizeFaces];
    	this.faces=new int[sizeFaces][];
    	
    	int i=0;
    	for(TriangulationDSVertex_2<Point_3> v: mesh.vertices) {
    		points[i]=v.getPoint();
    		v.index=i; // use indices in order to number the vertices, useful later
    		i++;
    	}
    	
    	for(i=0;i<sizeFaces;i++){
    		int d=3; // the mesh is triangulated
    		faceDegrees[i]=d;
    		this.faces[i]=new int[d];
    	}
    	
    	i=0;
      	for(TriangulationDSFace_2<Point_3> f: mesh.faces){
    		faces[i][0]=f.vertex(0).index;
    		faces[i][1]=f.vertex(1).index;
    		faces[i][2]=f.vertex(2).index;
    		i++;
    	}
    	System.out.println("done");
  }
	
    /**
     * Create a shared vertex representation from a polyhedron (half-edge data structure)
     */
	public SharedVertexRepresentation(Polyhedron_3<Point_3> polyhedron){
    	System.out.print("Creating a shared vertex representation from polyhedron...");
    	long startTime=System.nanoTime(), endTime; // for evaluating time performances

    	this.sizeVertices=polyhedron.sizeOfVertices();
    	this.sizeFaces=polyhedron.sizeOfFacets();            
    	this.sizeHalfedges=polyhedron.sizeOfHalfedges();
    	
    	this.points=new Point_3[sizeVertices];
    	this.faceDegrees=new int[sizeFaces];
    	this.faces=new int[sizeFaces][];
    	
    	// store vertex indices in a hash table (for efficient retrieval)
    	HashMap<Vertex<Point_3>,Integer> vertexIndices=new HashMap<Vertex<Point_3>, Integer>();
    	
    	int i=0;
    	for(Vertex<Point_3> v: polyhedron.vertices) {
    		points[i]=v.getPoint();
    		vertexIndices.put(v, i);
    		i++;
    	}
    	
    	i=0;
    	for(Face<Point_3> f: polyhedron.facets){
    		int d=f.degree();
    		faceDegrees[i]=d;
    		this.faces[i]=new int[d];
    		
    		Halfedge<Point_3> h=f.getEdge();
    		for(int j=0;j<d;j++) {
    			this.faces[i][j]=vertexIndices.get(h.getVertex());
    			h=h.getNext();
    		}
    		i++;
    	}
    	
    	/*i=0;
      	for(Face<Point_3> f: polyhedron.facets){
    		faces[i]=f.getVertexIndices(polyhedron);
    		i++;
    	}*/
      	
        endTime=System.nanoTime();
        double duration=(double)(endTime-startTime)/1000000000.;
    	System.out.println("done ("+duration+" seconds)");
  }
	
    /**
     * Store the representation in OFF format
     */
	public void writeOffFile(String filename) throws IOException {
    	// store vertex indices in map 
    	 BufferedWriter out = new BufferedWriter (new FileWriter(filename));
       	 out.write ("OFF\n");
       	 out.write (sizeVertices + " " + sizeFaces + " 0\n");  
    	 for (Point_3 p : points)
    		 out.write(p.getX() + " " + p.getY() + " " + p.getZ() + "\n");
    	 
    	 for (int i=0; i<sizeFaces; i++) {
    		 out.write(""+faceDegrees[i]);
    		 for (int j=0; j<faceDegrees[i]; j++)
    			 out.write (" " + faces[i][j]);
    			 out.write ("\n");
    	 }
    	 out.close();
	}
	
}
