package Jcg.mesh;

import java.util.*;

import Jcg.geometry.*;
import Jcg.polyhedron.*;

/**
 * @author Luca Castelli Aleardi (Ecole Polytechnique, 2014)
 * This class provides methods for building a polyhedral mesh (Polyhedron_3 class)
 */
public class MeshBuilder<X extends Point_> {
		
	/**
	 * Efficient construction of a Polyhedron (Half-edge DS) for a surface mesh (from a share vertex representation) <br>
	 * <br>
	 * Remark: it works for arbitrary polygonal meshes,possibly having boundaries
	 */
	public Polyhedron_3<X> createMesh(X[] points, int[] faceDegrees, int[][] faces) {   
		System.out.print("Building a (pointer based) halfedge representation of a surface mesh...");
		long startTime=System.nanoTime(), endTime; // for evaluating time performances

		int sizeHalfedges=0; // counts the number of half-edges (twice the sum of face degrees)
        int nInsertedHalfedges=0; // counts the number of inserted half-edges
        
        for(int i=0;i<faceDegrees.length;i++) {
        	sizeHalfedges=sizeHalfedges+faceDegrees[i];
        }

		// allows to efficient retrieving an inserted half-edge
    	HashMap<HalfedgePair, Halfedge<X>> insertedHalfedges=new HashMap(sizeHalfedges);
    	
    	Polyhedron_3<X> result=new Polyhedron_3<X>(points.length,sizeHalfedges,faceDegrees.length);
        
    	int i;
        for(i=0;i<points.length;i++){ // add geometric points
        	Vertex<X> v=new Vertex<X>(points[i]);
        	v.index=i;
            result.vertices.add(v); // adding a vertex
        }
        
        Face<X> _face;
        for(i=0;i<faceDegrees.length;i++) {
        	int d=faceDegrees[i]; // degree of face i
            _face=new Face<X>(); // creating a face
            result.facets.add(_face); // adding the created face
            
            // creating and storing the 'd' half-edges in a face
            Halfedge[] _edges=new Halfedge[d];
            for(int j=0;j<d;j++) {
            	Halfedge<X> e=new Halfedge<X>();
            	result.halfedges.add(e);
            	_edges[j]=e; // storing the edges of the face
            	
            	// setting incident (target) vertex
            	e.setVertex(result.vertices.get(faces[i][(j+1)%d]));
            	// half-edges must be incident to the current face
            	e.setFace(_face);
            }
            
            // setting incidence relations between half-edges in the same face
            for(int j=0;j<d;j++) {
            	int indNext=(j+1)%d; // index of next half-edge
            	int indPrev; // index of previous half-edge
            	if(j>0) indPrev=j-1;
            	else indPrev=d-1;
            	
            	_edges[j].setNext(_edges[indNext]);
            	_edges[j].setPrev(_edges[indPrev]);
            }
            
            // setting opposite half-edge
            for(int j=0;j<d;j++) {
            	HalfedgePair pair0=new HalfedgePair(faces[i][j], faces[i][(j+1)%d]);
            	if(insertedHalfedges.containsKey(pair0)==false) {
            		insertedHalfedges.put(pair0, _edges[j]);
            		//System.out.println("inserted edge ");
            		nInsertedHalfedges++;
            	}
            	else {
            		//System.out.println("edge already inserted");
            		Halfedge<X> eOpposite=insertedHalfedges.get(pair0);
            		_edges[j].setOpposite(eOpposite);
            		eOpposite.setOpposite(_edges[j]);
            	}
            }
            
            // setting an edge incident to the current face
            _face.setEdge(_edges[0]);
        }
        
        // setting the vertex incident to a half-edge
		for(Halfedge pedge: result.halfedges) {
        	Vertex<X> pvertex=pedge.vertex;
        	pvertex.setEdge(pedge);
        }
		
		// deal with boundary edges
		addBoundaryEdges(result);

		endTime=System.nanoTime();
        double duration=(double)(endTime-startTime)/1000000000.;
    	System.out.println("done ("+duration+" seconds)");
		
        return result;        
    }
	
	/**
	 * Efficient construction of a polyhedral mesh (Half-edge DS) 
	 * (from a share vertex representation)
	 */
	Polyhedron_3<X> createMeshOld(X[] points, int[] faceDegrees, int[][] faces) {   
		System.out.print("Building a (pointer based) halfedge representation of a triangle mesh...");
		
		int sizeHalfedges=0; // count the number of half-edges (twice the sum of face degrees)s
        int nInsertedHalfedges=0; // counts the number of inserted halfedges

        // allows to efficient retrieving an inserted halfedge
    	HashMap<HalfedgePair, Halfedge<X>> insertedHalfedges=new HashMap(sizeHalfedges);
    	
    	Polyhedron_3<X> result=new Polyhedron_3<X>(points.length,sizeHalfedges,faceDegrees.length);
        
    	int i;
    	for(i=0;i<points.length;i++){ // add geometric points
    		Vertex<X> v=new Vertex<X>(points[i]);
        	v.index=i;
            result.vertices.add(v); // adding a vertex
        }
        
        Face<X> _face;
        for(i=0;i<faceDegrees.length;i++) {         
            _face=new Face<X>(); // creating a face
            result.facets.add(_face); // adding the created face
            
            // creating and storing the three halfedges in a face
            Halfedge<X> e0=new Halfedge<X>();
            Halfedge<X> e1=new Halfedge<X>();
            Halfedge<X> e2=new Halfedge<X>();
            result.halfedges.add(e0);
            result.halfedges.add(e1);
            result.halfedges.add(e2);
            
            // setting incident (target vertex)
            e0.setVertex(result.vertices.get(faces[i][1]));
            e1.setVertex(result.vertices.get(faces[i][2]));
            e2.setVertex(result.vertices.get(faces[i][0]));
            
            // halfedges must be incident to the current face
            e0.setFace(_face);
            e1.setFace(_face);
            e2.setFace(_face);
            
            // incidence relations between halfedges in the same face
            e0.setNext(e1); e0.setPrev(e2);
            e1.setNext(e2); e1.setPrev(e0);
            e2.setNext(e0); e2.setPrev(e1);
            
            // setting first opposite halfedge
            HalfedgePair pair0=new HalfedgePair(faces[i][0], faces[i][1]);
            if(insertedHalfedges.containsKey(pair0)==false) {
            	insertedHalfedges.put(pair0, e0);
            	//System.out.println("inserted edge ");
            	nInsertedHalfedges++;
            }
            else {
            	//System.out.println("edge already inserted");
            	Halfedge<X> eOpposite=insertedHalfedges.get(pair0);
            	e0.setOpposite(eOpposite);
            	eOpposite.setOpposite(e0);
            }
            // setting second opposite halfedge
            HalfedgePair pair1=new HalfedgePair(faces[i][1], faces[i][2]);
            if(insertedHalfedges.containsKey(pair1)==false) {
            	insertedHalfedges.put(pair1, e1);
            	nInsertedHalfedges++;
            }
            else {
            	//System.out.println("edge already inserted");
            	Halfedge<X> eOpposite=insertedHalfedges.get(pair1);
            	e1.setOpposite(eOpposite);
            	eOpposite.setOpposite(e1);
            }
            // setting third opposite halfedge
            HalfedgePair pair2=new HalfedgePair(faces[i][2], faces[i][0]);
            if(insertedHalfedges.containsKey(pair2)==false) {
            	insertedHalfedges.put(pair2, e2);
            	nInsertedHalfedges++;
            }
            else {
            	//System.out.println("edge already inserted");
            	Halfedge<X> eOpposite=insertedHalfedges.get(pair2);
            	e2.setOpposite(eOpposite);
            	eOpposite.setOpposite(e2);
            }
            
            // setting an edge incident to the current face
            _face.setEdge(e0);
        }
        
        // setting the vertex incident to a halfedge
		for(i=0;i<result.halfedges.size();i++) {
        	Halfedge<X> pedge=(Halfedge<X>)result.halfedges.get(i);
        	Vertex<X> pvertex=pedge.vertex;
        	pvertex.setEdge(pedge);
        }
		System.out.println("done");
		
		// deal with boundary edges
		addBoundaryEdges(result);
		
        return result;        
    }

	/**
	 * Add and set boundary half-edges to the representation
	 * Remark: it allows to deal with meshes having boundaries
	 */
	private void addBoundaryEdges(Polyhedron_3<X> mesh) {
		//System.out.print("Processing boundaries of halfedge representation ...");
    	// store boundary edges (corresponding to half-edges without opposite reference)
    	HashSet<Halfedge<X>> isTreated=new HashSet<Halfedge<X>>(); // says whether a boundary edges has already been treated
    	LinkedList<Halfedge<X>> boundaryEdges=new LinkedList<Halfedge<X>>(); // stores boundary edges
        int nInsertedHalfedges=0; // counts the number of inserted halfedges
        
        for(Halfedge<X> e: mesh.halfedges) { // store all boundary edges
        	if(e.opposite==null)
        		boundaryEdges.add(e);
        }
        //System.out.println(boundaryEdges.size()+" boundary edges found");
        
        int bCount=0;
        while (boundaryEdges.isEmpty()==false) { // set opposite references of all boundary edges
        	Halfedge<X> firstEdge=boundaryEdges.poll(); // get a boundary
        	//System.out.println("processing edge "+printEdge(firstEdge));
        	if(isTreated.contains(firstEdge)==false) { // the edge belongs to a new boundary
        		//System.out.println("Processing boundary "+bCount);
        		Halfedge<X> opFirstEdge=new Halfedge<X>();
        		mesh.halfedges.add(opFirstEdge);
        		opFirstEdge.setOpposite(firstEdge);
        		firstEdge.setOpposite(opFirstEdge);
        		opFirstEdge.setVertex(firstEdge.getPrev().getVertex());
        		isTreated.add(firstEdge); // mark the half-edge as treated
        		
        		Halfedge<X> previousEdge=firstEdge.getOpposite(); // useful to set neighboring references
        		Halfedge<X> e=this.getNextBoundaryHalfedge(firstEdge);
        		while(e!=firstEdge) { // edge e turns around the boundary face
        			Halfedge<X> opE=new Halfedge<X>();
        			mesh.halfedges.add(opE);
        			opE.setOpposite(e);
        			e.setOpposite(opE);
        			opE.setVertex(e.getPrev().getVertex());
        			previousEdge.setPrev(opE);
        			opE.setNext(previousEdge);
        			
        			isTreated.add(e); // mark the half-edge as treated
        			previousEdge=opE;
        			e=this.getNextBoundaryHalfedge(e); // visit next boundary half-edge
        		}
        		previousEdge.setPrev(opFirstEdge);
        		opFirstEdge.setNext(previousEdge);
        		bCount++;
        	}
        }    
    }
	
	/**
	 * Given a boundary half-edge e, returns the next boundary half-edge around the same face
	 * Warning: the mesh is supposed to be manifold (boundary cycles are disjoint)
	 */
	private Halfedge<X> getNextBoundaryHalfedge(Halfedge<X> e) {
		//System.out.println("e:"+printEdge(e)+" ");
		if(e.getNext().getOpposite()==null) // the two edges belong to the same face
			return e.getNext();
		
		// turn around vertex v, starting from the half-edge pEdge
		Halfedge<X> pEdge=e.getNext();
		// two conditions are necessary (to stop the while loop)
		// during the first pass, all opposite references of boundary edges are null
		// once the pEdge come at the starting point (edge e), all opposite references are not null anymore
		// to stop the while, just test whether the corresponding face exists
		while(pEdge.getOpposite()!=null && pEdge.getOpposite().getFace()!=null) {
			//System.out.print(" pEdge:"+printEdge(pEdge));
			pEdge=pEdge.getOpposite().getNext();
		}
		//System.out.println("next boundary edge:"+printEdge(pEdge));
		return pEdge;
    }
	
	/**
	 * Print the edge (the index of its extremities)
	 * Useful for debugging
	 */
	private String printEdge(Halfedge<X> e) {
		if(e==null) return "null";
		return "("+e.getPrev().getVertex().index+" - "+e.getVertex().index+")";
    }
	
	/**
	 * Convert an array of 3D points to an array of 2D points (by orthogonal projection on the z=0 hyperplane)
	 */
    public static Point_2[] Point3DToPoint2D(Point_3[] points) {
    	Point_2[] result=new Point_2[points.length];
    	for(int i=0;i<points.length;i++) {
    		result[i]=new Point_2(points[i].getX().doubleValue(), points[i].getY().doubleValue());
    	}
    	return result;
    }
            
}
