package Jcg.mesh;

import java.awt.Color;
import java.util.List;

import Jcg.geometry.Point_;
import Jcg.polyhedron.*;
import Jcg.triangulations2D.TriangulationDSFace_2;
import Jcg.triangulations2D.TriangulationDSVertex_2;
import Jcg.triangulations2D.TriangulationDS_2;

/**
 * @author Luca Castelli Aleardi
 * 
 * This class provides input/output methods for dealing with
 * plane and surface meshes (stored in various formats, e.g.: OFF, Metis, ...)
 */
public class IO {

    /**
     * Output a 3D (surface) triangulation to OFF file <p>
     * Warning: only for triangle meshes (faces must be triangular)
     * 
     * @param mesh a triangulation (Half-edge representation)
     * 
     */
    public static void writePolyedronToOFF(Polyhedron_3 mesh, String filename){
    	System.out.print("Creating OFF file from Polyhedron...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int nVertices=mesh.vertices.size();
    	int nFaces=mesh.facets.size();
    	
    	// file header
    	Jcg.io.IO.println("OFF");
    	Jcg.io.IO.println(nVertices+" "+nFaces+" "+0);
    	    	
    	// writing vertex coordinates
    	int i=0;
    	for(Object o: mesh.vertices) { // iterate over all vertices
    		Vertex<Point_> v=(Vertex<Point_>)o;
    		v.index=i;
    		Point_ p=v.getPoint();
    		double x=0, y=0, z=0;
    		
    		if(p!=null && p.getCartesian(0)!=null && p.getCartesian(1)!=null) {
    			x=p.getCartesian(0).doubleValue();
    			y=p.getCartesian(1).doubleValue();
    		}
    		if(p!=null && p.dimension()==3 && p.getCartesian(2)!=null)
    			z=p.getCartesian(2).doubleValue();
    		
    		Jcg.io.IO.println(""+x+" "+y+" "+z);
    		i++;
    	}
    	
      	// write face/vertex incidence relations
    	for(Object o: mesh.facets){ // iterate over all faces
      		Face<Point_> currentFace=(Face<Point_>)o;
    		int index0=currentFace.getEdge().vertex.index;
    		int index1=currentFace.getEdge().getNext().vertex.index;
    		int index2=currentFace.getEdge().getNext().getNext().vertex.index;
    		Jcg.io.IO.println(3+" "+index0+" "+index1+" "+index2);
    	}
    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

    /**
     * Reverse the orientation of the input surface and output to OFF file. <br>
     * The orientation of the faces is reversed. <p>
     * 
     * Warnings: <br>
     * -) only for triangle meshes (faces must be triangular) <br>
     * -) the orientation of the input mesh remains unchanged
     * 
     * 
     * @param mesh  input triangulation (Half-edge representation)
     * 
     */
    public static void writeReversedPolyedronToOFF(Polyhedron_3 mesh, String filename){
    	System.out.print("Creating OFF file from Polyhedron...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int nVertices=mesh.vertices.size();
    	int nFaces=mesh.facets.size();
    	
    	// file header
    	Jcg.io.IO.println("OFF");
    	Jcg.io.IO.println(nVertices+" "+nFaces+" "+0);
    	    	
    	// writing vertex coordinates
    	int i=0;
    	for(Object o: mesh.vertices) { // iterate over all vertices
    		Vertex<Point_> v=(Vertex<Point_>)o;
    		v.index=i;
    		Point_ p=v.getPoint();
    		double x=0, y=0, z=0;
    		
    		if(p!=null && p.getCartesian(0)!=null && p.getCartesian(1)!=null) {
    			x=p.getCartesian(0).doubleValue();
    			y=p.getCartesian(1).doubleValue();
    		}
    		if(p!=null && p.dimension()==3 && p.getCartesian(2)!=null)
    			z=p.getCartesian(2).doubleValue();
    		
    		Jcg.io.IO.println(""+x+" "+y+" "+z);
    		i++;
    	}
    	
      	// write face/vertex incidence relations
    	for(Object o: mesh.facets){ // iterate over all faces
      		Face<Point_> currentFace=(Face<Point_>)o;
    		int index0=currentFace.getEdge().vertex.index;
    		int index1=currentFace.getEdge().getNext().vertex.index;
    		int index2=currentFace.getEdge().getNext().getNext().vertex.index;
    		Jcg.io.IO.println(3+" "+index2+" "+index1+" "+index0);
    	}
    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

    /**
     * Output a 3D (surface) triangulation to OFF file
     * 
     * @param mesh a triangulation (Half-edge representation)
     * @param edgeColors an array containing the colors of edges (one for each half-edge)
     * 
     * warning: only for triangle meshes (faces must be triangular)
     */
    public static void writePolyedronToOFF(Polyhedron_3 mesh, double[][] edgeColors, String filename){
    	System.out.print("Creating OFF file from Polyhedron...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int nVertices=mesh.vertices.size();
    	int nFaces=mesh.facets.size();
    	
    	// file header
    	Jcg.io.IO.println("OFF");
    	Jcg.io.IO.println(nVertices+" "+nFaces+" "+0);
    	    	
    	// writing vertex coordinates
    	int i=0;
    	for(Object o: mesh.vertices) { // iterate over all vertices
    		Vertex<Point_> v=(Vertex<Point_>)o;
    		v.index=i;
    		Point_ p=v.getPoint();
    		double x=0, y=0, z=0;
    		
    		if(p!=null && p.getCartesian(0)!=null && p.getCartesian(1)!=null)
    			x=p.getCartesian(0).doubleValue();
    		if(p!=null && p.dimension()==3 && p.getCartesian(2)!=null)
    			z=p.getCartesian(2).doubleValue();
    		
    		Jcg.io.IO.println(""+x+" "+y+" "+z);
    		i++;
    	}
    	
      	// write face/vertex incidence relations
    	for(Object o: mesh.facets){ // iterate over all faces
      		Face<Point_> currentFace=(Face<Point_>)o;
    		int index0=currentFace.getEdge().vertex.index;
    		int index1=currentFace.getEdge().getNext().vertex.index;
    		int index2=currentFace.getEdge().getNext().getNext().vertex.index;
    		Jcg.io.IO.println(3+" "+index0+" "+index1+" "+index2);
    	}
    	
    	if(edgeColors!=null) { // write edge colors (if defined) to OFF file
        	for(Object o: mesh.halfedges){ // iterate over all halfedges
          		Halfedge e=(Halfedge)o;
        		int index0=e.getVertex().index;
        		int index1=e.getOpposite().getVertex().index;
        		
        		int eIndex=e.index;
        		String edgeColor=""+edgeColors[eIndex][0]+" "+edgeColors[eIndex][1]+" "+edgeColors[eIndex][2]+" "+edgeColors[eIndex][3];
        		if(index0<index1) // each edge is written only once
        			Jcg.io.IO.println(2+" "+index0+" "+index1+" "+edgeColor);
        	}
    		
    	}
    	
    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

    /**
     * Output a 3D (surface) mesh to METIS file <p>
     * 
     * @param mesh a triangulation (Half-edge representation)
     * 
     */
    public static void writePolyedronToMETIS(Polyhedron_3 mesh, String filename){
    	System.out.print("Creating METIS mesh file from Polyhedron...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	
    	int nVertices=mesh.vertices.size();
    	
    	// file header
    	Jcg.io.IO.println(nVertices);
    	    	
    	// writing vertex coordinates
    	int i=0;
    	List<Vertex> vertices=mesh.vertices;
    	for(Vertex v: vertices) {
    		v.index=i;
    		i++;
    	}

    	// write face/vertex incidence relations
    	for(Vertex v: vertices) {
    		List<Halfedge> edges=v.getOutgoingHalfedges();
    		for(Halfedge e: edges) {
    			Jcg.io.IO.print(e.getVertex().index+1+" "); // warning: indices start from 1, in Metis format
    		}
    		Jcg.io.IO.println("");
    	}

    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

    /**
     * Output a 3D (surface) mesh as a list of EDGES (with vertices separated by commas) <p>
     * Indices of vertices start from 0
     * 
     * @param mesh a triangulation (Half-edge representation)
     * 
     */
    public static void writePolyedronToEDGES(Polyhedron_3 mesh, String filename, String separatingText){
    	System.out.print("Creating EDGES file from Polyhedron...");
    	Jcg.io.IO.writeNewTextFile(filename);
    	    	
    	// writing vertex coordinates
    	int i=0;
    	List<Vertex> vertices=mesh.vertices;
    	for(Vertex v: vertices) {
    		v.index=i;
    		i++;
    	}

    	// write face/vertex incidence relations
    	List<Halfedge> edges=mesh.halfedges;
    	for(Halfedge e: edges) {
    		if(e.getVertex().index>e.getOpposite().getVertex().index) // write edges only once
    			Jcg.io.IO.println(e.getOpposite().getVertex().index+separatingText+e.getVertex().index); // remark: indices start from 0
    	}

    	Jcg.io.IO.writeStandardOutput();
    	System.out.println("done ("+filename+")");
    }

}
