package Jcg.geometry.kernel;

import java.math.BigDecimal;

import Jcg.geometry.Algebra;
import Jcg.geometry.AlgebraExact;
import Jcg.geometry.Point_3;
import Jcg.geometry.Segment_3;
import Jcg.geometry.Triangle_3;

/**
 * Filtered geometric predicates for 3D objects (points, segments, rays)
 * 
 * @author Luca Castelli Aleardi and Steve Oudot (INF562, Ecole Polytechnique)
 * @version feb 2014
 */
public class FilteredPredicates_3 implements GeometricPredicates_3 {

	// Bounds for filters depend on the degree of the predicate
	private static final double epsilon3 = 1e-12;
	private static final double epsilon4 = 1e-9;
//	private static final double epsilon3 = Double.MAX_VALUE;
//	private static final double epsilon4 = Double.MAX_VALUE;

    /**
     * Intersection test between a segment and a triangle in 3D (test is filtered)
     *
     * @param s a 3D segment
     * @param t a 3D triangle
     * @return true whether s and t do intersect, false otherwise
     */
    public boolean doIntersect(Segment_3 s, Triangle_3 t) {
    	Point_3 p = (Point_3) s.source(), q = (Point_3) s.target();
    	Point_3 a = t.vertex(0), b = t.vertex(1), c = t.vertex(2);
    	int orientabcp = orientation(a,b,c,p);
    	int orientabcq = orientation(a,b,c,q);
    	int orientabpq = orientation(a,b,p,q);
    	int orientbcpq = orientation(b,c,p,q);
    	int orientcapq = orientation(c,a,p,q);
    	
    	// We are redundant between ab, bc and ca to avoid problems when line pq intersects one of the edges
    	return orientabcp*orientabcq <= 0 &&
    	orientabpq*orientbcpq >= 0 && orientbcpq*orientcapq >= 0 && orientcapq*orientabpq >= 0; 
    }

    /** 
     * Compute the orientation of tetrahedron (test is filtered)
     * 
     * @param a point 3D (one vertex of a tetrahedron)
     * @param b point 3D (one vertex of a tetrahedron)
     * @param c point 3D (one vertex of a tetrahedron)
     * @param d point 3D (one vertex of a tetrahedron)
     * @return  +1 if orientation is direct, -1 if orientation is indirect, 0 if points are coplanar
     */
    public int orientation(Point_3 a, Point_3 b, Point_3 c, Point_3 d) {
    	// Note: we multiply det by -1 because positive orientation <=> negative determinant in odd dimensions
    	double det = -Algebra.det44(new double[] {a.x, a.y, a.z, 1, b.x, b.y, b.z, 1, c.x, c.y, c.z, 1, d.x, d.y, d.z, 1});
    	if (det > epsilon3)
    		return 1;
    	else if (det < -epsilon3)
    		return -1;
    	
    	// else perform exact computation
    	BigDecimal ax = BigDecimal.valueOf(a.x);
    	BigDecimal ay = BigDecimal.valueOf(a.y);
    	BigDecimal az = BigDecimal.valueOf(a.z);
    	BigDecimal bx = BigDecimal.valueOf(b.x);
    	BigDecimal by = BigDecimal.valueOf(b.y);
    	BigDecimal bz = BigDecimal.valueOf(b.z);
    	BigDecimal cx = BigDecimal.valueOf(c.x);
    	BigDecimal cy = BigDecimal.valueOf(c.y);
    	BigDecimal cz = BigDecimal.valueOf(c.z);
    	BigDecimal dx = BigDecimal.valueOf(d.x);
    	BigDecimal dy = BigDecimal.valueOf(d.y);
    	BigDecimal dz = BigDecimal.valueOf(d.z);
    	
    	BigDecimal[] m=new BigDecimal[16];
    	m[0]=ax; m[1]=ay; m[2]=az; m[3]=BigDecimal.valueOf(1.0);
    	m[4]=bx; m[5]=by; m[6]=bz; m[7]=BigDecimal.valueOf(1.0);
    	m[8]=cx; m[9]=cy; m[10]=cz; m[11]=BigDecimal.valueOf(1.0);
    	m[12]=dx; m[13]=dy; m[14]=dz; m[15]=BigDecimal.valueOf(1.0);

    	// Note: we multiply det by -1 because positive orientation <=> negative determinant in odd dimensions
    	return AlgebraExact.det44(m).multiply(BigDecimal.valueOf(-1)).compareTo(BigDecimal.ZERO);
    }

}
