package Jcg.geometry.kernel;

import java.math.BigDecimal;

import Jcg.geometry.Algebra;
import Jcg.geometry.AlgebraExact;
import Jcg.geometry.Point_2;
import Jcg.geometry.Ray_2;
import Jcg.geometry.Segment_2;
import Jcg.geometry.Vector_2;

/**
 * Filtered geometric predicates for plane objects (points, segments, rays) <br>
 * <br>
 * Remark<br> Test is filtered: if precision bound <em>epsilon</em> is reached, then
 * perform computation with infinite precision arithmetic (using
 * the <tt>java.util.BigDecimal</tt> class)
 * 
 * @author Luca Castelli Aleardi and Steve Oudot (INF562, Ecole Polytechnique)
 * @version 2.0, updated 12/11/2022 (created, december 2022)
 */
public class FilteredPredicates_2 implements GeometricPredicates_2 {

	private static final double epsilon2 = 1e-15;
	private static final double epsilon3 = 1e-12;
   
    /** 
     * 
     * Returns the orientation of a, b and c
    *
    * @param a,b,c the 3 points to test
    * @return 1 if sequence (a, b, c) turns in counter-clockwise direction, -1 if it turns in clockwise direction, 0 if the points are collinear
    *
    * Test is filtered: if precision bound epsilon is reached, then
    * perform computation with infinite precision arithmetic (using
    * the java.util.BigDecimal class)
    */
    public int orientation (Point_2 a, Point_2 b, Point_2 c) {
    	double det = Algebra.det33(new double[] {a.x, a.y, 1, b.x, b.y, 1, c.x, c.y, 1});
    	if (det > epsilon2)
    		return 1;
    	else if (det < -epsilon2)
    		return -1;

    	// else perform exact computation
    	BigDecimal ax = BigDecimal.valueOf(a.x);
    	BigDecimal ay = BigDecimal.valueOf(a.y);
    	BigDecimal bx = BigDecimal.valueOf(b.x);
    	BigDecimal by = BigDecimal.valueOf(b.y);
    	BigDecimal cx = BigDecimal.valueOf(c.x);
    	BigDecimal cy = BigDecimal.valueOf(c.y);

    	return AlgebraExact.det33 (new BigDecimal[] {ax, ay, BigDecimal.valueOf(1),
    			bx, by, BigDecimal.valueOf(1), 
    			cx, cy, BigDecimal.valueOf(1)}).compareTo(BigDecimal.ZERO);
    }
    
    /** Returns true if a, b and c turn in counter-clockwise direction <br>
     * <br>
     * Remark<br> Test is filtered: if precision bound <em>epsilon</em> is reached, then
     * perform computation with infinite precision arithmetic (using
     * the <tt>java.util.BigDecimal</tt> class)
     * 
     * @param a,b,c the 3 points to test
     * @return true if a, b and c turn in counter-clockwise direction <br>
     * Warning: the result is false if the three points are collinear
     */
    public boolean isCounterClockwise(Point_2 a, Point_2 b, Point_2 c) {
    	return orientation(a, b, c) > 0;
    }
    
    /** Returns true if a, b and c turn in clockwise direction <br>
     * <br>
     * Remark<br> Test is filtered: if precision bound <em>epsilon</em> is reached, then
     * perform computation with infinite precision arithmetic (using
     * the <tt>java.util.BigDecimal</tt> class)
     * 
     * @param a,b,c the 3 points to test
     * @return true if a, b and c turn in clockwise direction <br>
     * Warning: the result is false if the three points are collinear
     */
    public boolean isClockwise(Point_2 a, Point_2 b, Point_2 c) {
    	return orientation(a, b, c) < 0;
    }
    
    /** Returns true if a, b and c lie on a same line <br>
     * <br>
     * Remark<br> Test is filtered: if precision bound <em>epsilon</em> is reached, then
     * perform computation with infinite precision arithmetic (using
     * the <tt>java.util.BigDecimal</tt> class)
     * 
     * @param a,b,c the 3 points to test
    * @return true if a, b and c are collinear (lie on a same line)
    */
   public boolean collinear(Point_2 a, Point_2 b, Point_2 c) {
	   return orientation(a, b, c) == 0;
   }

   /**
     * Tests if point p lies inside the circumcircle of triangle a,b,c
     *
     * @param a,b,c triangle
     * @param p point to test
     * @return  true/false
     *
     * Test is filtered: if precision bound epsilon is reached, then
     * perform computation with infinite precision arithmetic (using
     * the java.util.BigDecimal class)
     */
    public boolean inCircle(Point_2 p, Point_2 a, Point_2 b, Point_2 c) {
	double a2 = a.x*a.x + a.y*a.y;
	double b2 = b.x*b.x + b.y*b.y;
	double c2 = c.x*c.x + c.y*c.y;
	double p2 = p.x*p.x + p.y*p.y;
	double det44 = Algebra.det44 (new double[] {a.x, a.y, a2, 1, b.x, b.y, b2, 1, c.x, c.y, c2, 1, p.x, p.y, p2, 1});

	if (det44 < -epsilon3) 
	    return true;
	else if (det44 > epsilon3)
		return false;

	// else use exact computation
	BigDecimal ax = BigDecimal.valueOf(a.x);
	BigDecimal ay = BigDecimal.valueOf(a.y);
	BigDecimal bx = BigDecimal.valueOf(b.x);
	BigDecimal by = BigDecimal.valueOf(b.y);
	BigDecimal cx = BigDecimal.valueOf(c.x);
	BigDecimal cy = BigDecimal.valueOf(c.y);
	BigDecimal px = BigDecimal.valueOf(p.x);
	BigDecimal py = BigDecimal.valueOf(p.y);
	BigDecimal ba2 = ax.multiply(ax).add(ay.multiply(ay));
	BigDecimal bb2 = bx.multiply(bx).add(by.multiply(by));
	BigDecimal bc2 = cx.multiply(cx).add(cy.multiply(cy));
	BigDecimal bp2 = px.multiply(px).add(py.multiply(py));
	return AlgebraExact.det44(new BigDecimal[] {ax, ay, ba2, BigDecimal.valueOf(1),
			bx, by, bb2, BigDecimal.valueOf(1),
			cx, cy, bc2, BigDecimal.valueOf(1),
			px, py, bp2, BigDecimal.valueOf(1)}).compareTo(BigDecimal.ZERO) <= 0;
	}


    /**
    * Returns true if segments s and t intersect
    * @param s,t the 2 segments
    * @return true if s,t intersect each other
    *
    * Test is filtered: if precision bound epsilon is reached, then
    * perform computation with infinite precision arithmetic (using
    * the java.util.BigDecimal class)
    */
   public boolean doIntersect(Segment_2 s, Segment_2 t) {
	   Point_2[] p = new Point_2[] {(Point_2)s.source(), (Point_2)s.target(), (Point_2)t.source(), (Point_2)t.target()};
	   if (liesOn(p[0], p[2], p[3]) || liesOn(p[1], p[2], p[3]) || liesOn(p[2], p[0], p[1]) || liesOn(p[3], p[0], p[1]))
		   return true;
	   else
		   return orientation(p[0], p[2], p[3]) * orientation(p[1], p[2], p[3]) < 0 &&
		   	orientation(p[2], p[0], p[1]) * orientation(p[3], p[0], p[1]) < 0;
   }


  /**
   * Returns true if segment s and ray r intersect
   * Test is not filtered.
   * @param s the segment
   * @param r the ray
   * @return true if s intersects r
   */
  public boolean doIntersect(Segment_2 s, Ray_2 r) {
	  Vector_2 pa = new Vector_2((Point_2)r.source(), (Point_2)s.source());
	  Vector_2 pb = new Vector_2((Point_2)r.source(), (Point_2)s.target());
	  double sqNorm = 1+Math.max(pa.squaredLength().doubleValue(), 
			  					pb.squaredLength().doubleValue());
	  return doIntersect(s, new Segment_2
			  (r.source(), 
					  r.source().sum(r.direction().multiplyByScalar(sqNorm))));
  } 
  
  /** 
     * Returns true if point p lies on segment ab
     *
     * @param a,b,p the 3 points
     * @return true if ab contains point p
     */    
 public boolean liesOn(Point_2 p, Point_2 a, Point_2 b) {    
	 /* If ab not vertical, check betweenness on x; else on y. */
	 if(!collinear(p,a,b))
		 return false;
	 if ( !a.x.equals(b.x) )
		 return ((a.x <= p.x) && (p.x <= b.x)) ||
		 ((a.x >= p.x) && (p.x >= b.x));
	 else
		 return ((a.y <= p.y) && (p.y <= b.y)) ||
		 ((a.y >= p.y) && (p.y >= b.y));
 }

}
