package Jcg.geometry.kernel;

import java.math.BigDecimal;

import Jcg.geometry.AlgebraExact;
import Jcg.geometry.Point_2;
import Jcg.geometry.Ray_2;
import Jcg.geometry.Segment_2;
import Jcg.geometry.Vector_2;

/**
 * Exact geometric predicates for plane objects (points, segments, rays)
 * Computations are performed with infinite precision arithmetic 
 * (using the java.util.BigDecimal class)
 * 
 * @author Luca Castelli Aleardi and Steve Oudot (INF562, Ecole Polytechnique)
 * @version dec 2012
 */
public class ExactPredicates_2 implements GeometricPredicates_2 {

    /** 
     * Returns the orientation of a, b and c
    *
    * @param a,b,c the 3 points to test
    * @return 	1 if sequence (a, b, c) turns in counter-clockwise direction, 
    * 			-1 if it turns in clockwise direction, 
    * 			0 if the points are collinear
    *
    */
    public int orientation (Point_2 a, Point_2 b, Point_2 c) {
    	BigDecimal ax = BigDecimal.valueOf(a.x);
    	BigDecimal ay = BigDecimal.valueOf(a.y);
    	BigDecimal bx = BigDecimal.valueOf(b.x);
    	BigDecimal by = BigDecimal.valueOf(b.y);
    	BigDecimal cx = BigDecimal.valueOf(c.x);
    	BigDecimal cy = BigDecimal.valueOf(c.y);

    	return AlgebraExact.det33 (new BigDecimal[] {ax, ay, BigDecimal.valueOf(1),
    			bx, by, BigDecimal.valueOf(1), 
    			cx, cy, BigDecimal.valueOf(1)}).compareTo(BigDecimal.ZERO);
    }
    
    /** 
     * Returns true if a, b and c turn in counter-clockwise direction
     *
     * @param a,b,c the 3 points to test
     * @return true if a, b and c turn in counter-clockwise direction. <br>
     * Warning: the result is false if the three points are collinear
     *
     */
    public boolean isCounterClockwise(Point_2 a, Point_2 b, Point_2 c) {
    	return orientation(a, b, c) > 0;
    }
    
    /** 
     * Returns true if a, b and c turn in clockwise direction
     *
     * @param a,b,c the 3 points to test
     * @return true if a, b and c turn in clockwise direction. <br>
     * Warning: the result is false if the three points are collinear
     *
     */
    public boolean isClockwise(Point_2 a, Point_2 b, Point_2 c) {
    	return orientation(a, b, c) < 0;
    }
    
    /** Returns true if a, b and c lie on a same line
    *
    * @param a,b,c the 3 points to test
    * @return true if a, b and c are collinear (lie on a same line)
    *
    * Test is filtered: if precision bound epsilon is reached, then
    * perform computation with infinite precision arithmetic (using
    * the java.util.BigDecimal class)
    */
   public boolean collinear(Point_2 a, Point_2 b, Point_2 c) {
	   return orientation(a, b, c) == 0;
   }

   /**
     * Tests if point p lies inside the circumcircle of triangle a,b,c
     *
     * @param a,b,c triangle
     * @param p point to test
     * @return  true/false
     *
     */
    public boolean inCircle(Point_2 p, Point_2 a, Point_2 b, Point_2 c) {
	BigDecimal ax = BigDecimal.valueOf(a.x);
	BigDecimal ay = BigDecimal.valueOf(a.y);
	BigDecimal bx = BigDecimal.valueOf(b.x);
	BigDecimal by = BigDecimal.valueOf(b.y);
	BigDecimal cx = BigDecimal.valueOf(c.x);
	BigDecimal cy = BigDecimal.valueOf(c.y);
	BigDecimal px = BigDecimal.valueOf(p.x);
	BigDecimal py = BigDecimal.valueOf(p.y);
	BigDecimal ba2 = ax.multiply(ax).add(ay.multiply(ay));
	BigDecimal bb2 = bx.multiply(bx).add(by.multiply(by));
	BigDecimal bc2 = cx.multiply(cx).add(cy.multiply(cy));
	BigDecimal bp2 = px.multiply(px).add(py.multiply(py));
	return AlgebraExact.det44(new BigDecimal[] {ax, ay, ba2, BigDecimal.valueOf(1),
			bx, by, bb2, BigDecimal.valueOf(1),
			cx, cy, bc2, BigDecimal.valueOf(1),
			px, py, bp2, BigDecimal.valueOf(1)}).compareTo(BigDecimal.ZERO) <= 0;
	}


    /**
    * Returns true if segments s and t intersect
    * @param s,t the 2 segments
    * @return true if s,t intersect each other
    *
    */
   public boolean doIntersect(Segment_2 s, Segment_2 t) {
	   Point_2[] p = new Point_2[] {(Point_2)s.source(), (Point_2)s.target(), (Point_2)t.source(), (Point_2)t.target()};
	   if (liesOn(p[0], p[2], p[3]) || liesOn(p[1], p[2], p[3]) || liesOn(p[2], p[0], p[1]) || liesOn(p[3], p[0], p[1]))
		   return true;
	   else
		   return orientation(p[0], p[2], p[3]) * orientation(p[1], p[2], p[3]) < 0 &&
		   	orientation(p[2], p[0], p[1]) * orientation(p[3], p[0], p[1]) < 0;
   }


  /** 
   * Returns true if segment s and ray r intersect
   *
   * @param s the segment
   * @param r the ray
   * @return true if s intersects r
   *
   * Warning: test is not exact.
   */
  public boolean doIntersect(Segment_2 s, Ray_2 r) {
	  Vector_2 pa = new Vector_2((Point_2)r.source(), (Point_2)s.source());
	  Vector_2 pb = new Vector_2((Point_2)r.source(), (Point_2)s.target());
	  double sqNorm = 1+Math.max(pa.squaredLength().doubleValue(), 
			  					pb.squaredLength().doubleValue());
	  return doIntersect(s, new Segment_2
			  (r.source(), 
					  r.source().sum(r.direction().multiplyByScalar(sqNorm))));
  } 
  
  /** 
     * Returns true if point p lies on segment ab
     *
     * @param a,b,p the 3 points
     * @return true if ab contains point p
     */    
 public boolean liesOn(Point_2 p, Point_2 a, Point_2 b) {    
	 /* If ab not vertical, check betweenness on x; else on y. */
	 if(!collinear(p,a,b))
		 return false;
	 if ( !a.x.equals(b.x) )
		 return ((a.x <= p.x) && (p.x <= b.x)) ||
		 ((a.x >= p.x) && (p.x >= b.x));
	 else
		 return ((a.y <= p.y) && (p.y <= b.y)) ||
		 ((a.y >= p.y) && (p.y >= b.y));
 }

}
