package Jcg.geometry.kernel;

import Jcg.geometry.Algebra;
import Jcg.geometry.Point_2;
import Jcg.geometry.Ray_2;
import Jcg.geometry.Segment_2;
import Jcg.geometry.Vector_2;

/**
 * Approximate geometric predicates for plane objects (points, segments, rays)
 * 
 * @author Luca Castelli Aleardi and Steve Oudot (INF562, Ecole Polytechnique)
 * @version dec 2012
 */
public class ApproximatePredicates_2 implements GeometricPredicates_2 {

    /** 
     * 
     * Returns the orientation of a, b and c
    *
    * @param a,b,c the 3 points to test
    * @return 1 if sequence (a, b, c) turns in counter-clockwise direction, -1 if it turns in clockwise direction, 0 if the points are collinear
    *
    */
    public int orientation (Point_2 a, Point_2 b, Point_2 c) {
    	double det = Algebra.det33(new double[] {a.x, a.y, 1, b.x, b.y, 1, c.x, c.y, 1});
    	if (det > 0)
    		return 1;
    	else if (det < 0)
    		return -1;
    	else
    		return 0;
    }
    
    /** Returns true if a, b and c turn in counter-clockwise direction
     *
     * @param a,b,c the 3 points to test
     * @return true if a, b and c turn in counter-clockwise direction. <br>
     * Warning: the result is false if the three points are collinear
     *
     */
    public boolean isCounterClockwise(Point_2 a, Point_2 b, Point_2 c) {
    	return orientation(a, b, c) > 0;
    }
    
    /** Returns true if a, b and c turn in clockwise direction
     *
     * @param a,b,c the 3 points to test
     * @return true if a, b and c turn in clockwise direction. <br>
     * Warning: the result is false if the three points are collinear
     *
     */
    public boolean isClockwise(Point_2 a, Point_2 b, Point_2 c) {
    	return orientation(a, b, c) < 0;
    }
    
    /** Returns true if a, b and c lie on a same line
    *
    * @param a,b,c the 3 points to test
    * @return true if a, b and c are collinear (lie on a same line)
    *
    */
   public boolean collinear(Point_2 a, Point_2 b, Point_2 c) {
	   return orientation(a, b, c) == 0;
   }

   /**
     * Tests if point p lies inside the circumcircle of triangle a,b,c
     *
     * @param a,b,c triangle
     * @param p point to test
     * @return  true/false
     *
     */
    public boolean inCircle(Point_2 p, Point_2 a, Point_2 b, Point_2 c) {
    	double a2 = a.x*a.x + a.y*a.y;
    	double b2 = b.x*b.x + b.y*b.y;
    	double c2 = c.x*c.x + c.y*c.y;
    	double p2 = p.x*p.x + p.y*p.y;
    	double det44 = Algebra.det44 (new double[] {a.x, a.y, a2, 1, b.x, b.y, b2, 1, c.x, c.y, c2, 1, p.x, p.y, p2, 1});

    	if (det44 <=0) 
    		return true;
    	else
    		return false;
	}


    /**
    * Returns true if segments 's' and 't' intersect
    * @param s   	first segment
    * @param t   	second segment
    * @return true 	if s and t intersect each other
    *
    */
   public boolean doIntersect(Segment_2 s, Segment_2 t) {
	   Point_2[] p = new Point_2[] {(Point_2)s.source(), (Point_2)s.target(), (Point_2)t.source(), (Point_2)t.target()};
	   if (liesOn(p[0], p[2], p[3]) || liesOn(p[1], p[2], p[3]) || liesOn(p[2], p[0], p[1]) || liesOn(p[3], p[0], p[1]))
		   return true;
	   else
		   return orientation(p[0], p[2], p[3]) * orientation(p[1], p[2], p[3]) < 0 &&
		   	orientation(p[2], p[0], p[1]) * orientation(p[3], p[0], p[1]) < 0;
   }


  /** 
   * Returns true if segment s and ray r intersect
   *
   * @param s the segment
   * @param r the ray
   *
   * @return true if s intersects r
   */
  public boolean doIntersect(Segment_2 s, Ray_2 r) {
	  Vector_2 pa = new Vector_2((Point_2)r.source(), (Point_2)s.source());
	  Vector_2 pb = new Vector_2((Point_2)r.source(), (Point_2)s.target());
	  double sqNorm = 1+Math.max(pa.squaredLength().doubleValue(), 
			  					pb.squaredLength().doubleValue());
	  return doIntersect(s, new Segment_2
			  (r.source(), 
					  r.source().sum(r.direction().multiplyByScalar(sqNorm))));
  }
  
  /** 
     * Returns true if point p lies on segment ab
     *
     * @param a,b,p the 3 points
     * @return true if ab contains point p
     */    
 public boolean liesOn(Point_2 p, Point_2 a, Point_2 b) {    
	 /* If ab not vertical, check betweenness on x; else on y. */
	 if(!collinear(p,a,b))
		 return false;
	 if ( !a.x.equals(b.x) )
		 return ((a.x <= p.x) && (p.x <= b.x)) ||
		 ((a.x >= p.x) && (p.x >= b.x));
	 else
		 return ((a.y <= p.y) && (p.y <= b.y)) ||
		 ((a.y >= p.y) && (p.y >= b.y));
 }

}
