package Jcg.geometry.kernel;

import Jcg.geometry.Algebra;
import Jcg.geometry.Point_2;
import Jcg.geometry.Ray_2;
import Jcg.geometry.Segment_2;
import Jcg.geometry.Vector_2;

/**
 * Approximate geometric constructions for plane objects (points, segments, rays)
 *
 * @author Luca Castelli Aleardi and Steve Oudot, Ecole Polytechnique (INF562)
 * @version dec 2012
 */
public class ApproximateConstructions_2 implements GeometricConstructions_2 {

    /**
     * Returns the (approximate) circumcenter of input triangle
     *
     * @param p0,p1,p2 the three vertices of the triangle
     *
     */
    public Point_2 circumCenter (Point_2 p0, Point_2 p1, Point_2 p2) {
    	double ex=p1.x-p0.x, ey=p1.y-p0.y;
    	double nx=p2.y-p1.y, ny=p1.x-p2.x;
    	double dx=(p0.x-p2.x)*0.5, dy=(p0.y-p2.y)*0.5;
    	double s=(ex*dx+ey*dy)/(ex*nx+ey*ny);
    	double cx=(p1.x+p2.x)*0.5+s*nx;
    	double cy=(p1.y+p2.y)*0.5+s*ny;
    	return new Point_2( cx, cy );
    }
    
    /**
     * Returns the distance (non exact computation) of a point to a segment
     *
     * @param ps,pe the start/end of the segment
     * @param p the query point
     * @return the distance of p to [ps,pe]
     */
    public Number distanceToSegment(Point_2 ps, Point_2 pe, Point_2 p) {
    	if (ps.x==pe.x && ps.y==pe.y) return ps.squareDistance(p);
    	
    	double sx=pe.x-ps.x;
    	double sy=pe.y-ps.y;
    	
    	double ux=p.x-ps.x;
    	double uy=p.y-ps.y;
    	
    	double dp=sx*ux+sy*uy;
    	if (dp<0) return ps.squareDistance(p);
    	
    	double sn = sx*sx+sy*sy;
    	if (dp>sn) return pe.squareDistance(p);
    	
    	double ratio = (double)dp/sn;
    	double projx = (ps.x + ratio*sx);
    	double projy = (ps.y + ratio*sy);
    	
    	return p.squareDistance(new Point_2(projx,projy));
    }
    
    /** 
     * Returns approximate intersection between segments
     * 
     * @param s,t the 2 segments
     * @return approximate intersection point of s,t
     */
    public Point_2 intersect(Segment_2 s, Segment_2 t) {
  		double ax = ((Point_2)s.source()).getX().doubleValue();  
  		double ay = ((Point_2)s.source()).getY().doubleValue();  
  		double bx = ((Point_2)s.target()).getX().doubleValue();  
  		double by = ((Point_2)s.target()).getY().doubleValue();  
  		double cx = ((Point_2)t.source()).getX().doubleValue();  
  		double cy = ((Point_2)t.source()).getY().doubleValue();  
  		double dx = ((Point_2)t.target()).getX().doubleValue();  
  		double dy = ((Point_2)t.target()).getY().doubleValue();  
  		double det = Algebra.det22(bx-ax, by-ay, dx-cx, dy-cy);
  		double alpha = Algebra.det22(dx-cx, dy-cy, dx-bx, dy-by) / det;
  		return new Point_2 (alpha*ax + (1-alpha)*bx, alpha*ay + (1-alpha)*by);
    }
    
    /** 
     * Return approximate intersection between segment s and ray r
     *
     * @param s the segment
     * @param r the ray
     * @return the intersection of s and r
     *
     */
    public Point_2 intersect(Segment_2 s, Ray_2 r) {
   	  Vector_2 pa = new Vector_2((Point_2)r.source(), (Point_2)s.source());
   	  Vector_2 pb = new Vector_2((Point_2)r.source(), (Point_2)s.target());
   	  double sqNorm = 1+Math.max(pa.squaredLength().doubleValue(), 
   			  					pb.squaredLength().doubleValue());
   	  return intersect(s, new Segment_2
   			  (r.source(), 
   					  r.source().sum(r.direction().multiplyByScalar(sqNorm))));
    }

}
