package Jcg.geometry;
/**
 * A class for representing a 2D points with rational coordinates
 * 
 * @author Luca Castelli Aleardi (Ecole Polytechnique, dec 2022)
 *
 */
public class RationalPoint_2 implements Point_ {
	/** geometric coordinates of the points (rational numbers) */
	public Rational x,y;

	/** integer index of the point: useful in some applications */
	public int index=-1;

	public RationalPoint_2() {}

	/**
	 * Initialize a 2D point with rational coordinates. <br>
	 * <br>
	 * Warning: this constructor does not define the index of the point
	 * 
	 * @param x  		x-coordinate
	 * @param y  		y-coordinate
	 */
	public RationalPoint_2(Rational x, Rational y) { 
		this.x=x; 
		this.y=y;
		this.index=-1; // index not defined
	}

	/**
	 * Initialize a rational 2D point, with integer coordinates. <br>
	 * <br>
	 * Warning: this constructor does not define the index of the point
	 * 
	 * @param x  		x-coordinate
	 * @param y  		y-coordinate
	 */
	public RationalPoint_2(int x, int y) { 
		this.x=new Rational(x, 1); 
		this.y=new Rational(y, 1);
		this.index=-1; // index not defined
	}

	/**
	 * Initialize a rational 2D point equal to an integer point <tt>p</tt>. <br>
	 * <br>
	 * Warning: this constructor does not define the index of the point
	 * 
	 * @param p  	integer 2D point
	 */
	public RationalPoint_2(GridPoint_2 p) { 
		this.x=new Rational(p.getX(), 1); 
		this.y=new Rational(p.getY(), 1);
		this.index=-1; // index not defined
	}

	/**
	 * Initialize a 2D points with rational coordinates, and an <em>index</em>
	 * 
	 * @param x  		x-coordinate
	 * @param y  		y-coordinate
	 * @param index  	index (useful for indexing points in some applications)
	 */
	public RationalPoint_2(Rational x,Rational y, int index) { 
		this.x=x; 
		this.y=y;
		this.index=-1; // index not defined
	}

	/** 
	 * Make a copy of the input point <tt>p</tt> <br>
	 * <br>
	 * Warning: <tt>p</tt> must be a <tt>Rational_2</tt> point
	 * 
	 * @param p  input point to copy (<tt>p</tt> must be a <tt>Rational_2</tt> point)
	 **/
	public RationalPoint_2(RationalPoint_2 p) {
		this.x=(Rational)p.getCartesian(0); 
		this.y=(Rational)p.getCartesian(1); 
	}

	/** 
	 * Make a copy of the input point <tt>p</tt> <br>
	 * <br>
	 * Warning: <tt>p</tt> must be a <tt>Rational_2</tt> point
	 * 
	 * @param p  input point to copy (<tt>p</tt> must be a <tt>Rational_2</tt> point)
	 **/
	public RationalPoint_2(Point_ p) {
		if(p instanceof RationalPoint_2) {
			this.x=(Rational)p.getCartesian(0); 
			this.y=(Rational)p.getCartesian(1); 
		}
		else if(p instanceof GridPoint_2) {
			GridPoint_2 o=(GridPoint_2)p;
			this.x=new Rational(o.getX(), 1); 
			this.y=new Rational(o.getY(), 1);
		}
		else
			throw new Error("Wrong conversion to RationalPoint_2");
	}

	/** Return the index */
	public int getIndex() {return index; }
	/** Return the x-coordinate */
	public Rational getX() {return x; }
	/** Return the y-coordinate */
	public Rational getY() {return y; }

	/** Set the index */
	public void setIndex(int index) {this.index=index; }
	
	/** 
	 * Set the x-coordinate of the current point <br>
	 * <br>
	 * Remark: the number <tt>x</tt> must be either <br>
	 * -) an <tt>Integer</tt> number <br>
	 * -) or a <tt>Rational</tt> number
	 **/
	public void setX(Number x) {
		if (x instanceof Rational) {
			Rational coord = (Rational) x;
			this.x.setRational(coord.numerator(), coord.denominator());
			return;
		}
		else if (x instanceof Integer) {
			Integer coord = (Integer) x;
			this.x=new Rational(coord.intValue(), 1);
			return;
		}
		throw new RuntimeException ("Wrong type number: method setX() is processing an object of type " + x.getClass());  	
	}

	/** 
	 * Set the y-coordinate of the current point <br>
	 * <br>
	 * Remark: the number <tt>y</tt> must be either <br>
	 * -) an integer number <br>
	 * -) or a <tt>Rational</tt> number
	 **/
	public void setY(Number y) {
		if (y instanceof Rational) {
			Rational coord = (Rational) y;
			this.y.setRational(coord.numerator(), coord.denominator());
			return;
		}
		else if (y instanceof Integer) {
			Integer coord = (Integer) y;
			this.y=new Rational(coord.intValue(), 1);
			return;
		}
		throw new RuntimeException ("Wrong type number: method setY() is processing an object of type " + x.getClass());  	
	}

	/** 
	 * Translate the current point of the vector <tt>o</tt> 
	 **/
	public void translateOf(Vector_ o) {
		throw new RuntimeException ("Not implemented yet");  	
	}

	/** 
	 * Check whether the current point is equal to point 'o'. <br>
	 * <br>
	 * Warning: point 'o' must be of type <tt>RationalPoint_2</tt>
	 **/
	public boolean equals(Object o) {
		if (o instanceof RationalPoint_2) {
			RationalPoint_2 p = (RationalPoint_2) o;
			return this.x.equals(p.getX()) && this.y.equals(p.getY()); 
		}
		else if (o instanceof GridPoint_2) {
			RationalPoint_2 p = new RationalPoint_2((GridPoint_2)o);
			return this.x.equals(p.getX()) && this.y.equals(p.getY()); 
		}
		throw new RuntimeException ("Method equals: comparing RationalPoint_2 with object of type " + o.getClass());  	
	}

	/**
	 * Warning: the result is very bad for rational points in [0..1]
	 */
	public int hashCode () {
		int maxValue=10000;
		int h=(int)((int)this.x.doubleValue()*(int)this.x.doubleValue())%maxValue;
		h=h+(int)this.y.doubleValue();
		//int h=(int)(this.x.doubleValue()*this.y.doubleValue());
		return h;
	}

	public double distanceFrom(RationalPoint_2 p) {
		return Math.sqrt(this.squareDistance(p).doubleValue());
	}

	/**
	 * Return the square of the Euclidean distance between the current point 'this' 
	 * and a point <tt>o</tt>
	 */
	public Rational squareDistance(Point_ o) {
		RationalPoint_2 p=null;
		if (o instanceof RationalPoint_2) {
			p=(RationalPoint_2)o;
		}
		else if(o instanceof GridPoint_2) {
			GridPoint_2 q=(GridPoint_2)o;
			p=new RationalPoint_2(new Rational(q.getX(), 1), new Rational(q.getY(), 1));
		}

		if(p==null)
			throw new Error("Error: the input point o is not Rational neither Integer");

		Rational dX=p.getX().subtract(this.getX());
		Rational dY=p.getY().subtract(this.getY());
		Rational prodX=dX.multiply(dX);
		Rational prodY=dY.multiply(dY);
		Rational result=prodX.add(prodY);
		return result;
	}

	/**
	 * Set the current point as the barycenter of an array of points
	 */	  
	public void barycenter(Point_[] points) {
		throw new Error("Not implemented yet");
	}

	/**
	 * Set the current point to be the linear combination of the input <tt>points</tt>,
	 * with respect to the given input <tt>coefficients</tt>
	 */
	public void linearCombination(Point_[] points, Number[] coefficients) {
		throw new Error("Not implemented yet");
	}

	/**
	 * Return a new rational point defined by the linear combination of the input <tt>points</tt>,
	 * with respect to the given input <tt>coefficients</tt>
	 */
	public static RationalPoint_2 linearCombination(RationalPoint_2 [] points, Number[] coefficients) {
		if (coefficients instanceof Rational[]) {
			Rational x=new Rational(0, 1),y=new Rational(0,1);

			for(int i=0;i<points.length;i++) { // compute linear combination
				Rational coeff=(Rational)coefficients[i];

				Rational x_=points[i].getX().multiply(coeff);
				x=x.add(x_);
				Rational y_=points[i].getY().multiply(coeff);
				y=y.add(y_);
			}
			return new RationalPoint_2(x,y);
		}
		throw new RuntimeException ("Method linearCombination: calculations with with object of type " + coefficients.getClass());  	
	}

	public String toString() {return "("+x+","+y+")"; }
	public int dimension() { return 2;}

	/** 
	 * Return the d-th coordinate of the point <br>
	 * 
	 * @param dim  the d-th coordinate of the point: <br>
	 * dim=0 for x-coordinate, dim=1 for y-coordinate
	 */
	public Rational getCartesian(int dim) {
		if(dim==0) return this.x;
		return this.y;
	}

	/** 
	 * Set the d-th coordinate of the point 
	 * 
	 * @param dim  the d-th coordinate of the point: <br>
	 * dim=0 for x-coordinate, dim=1 for y-coordinate
	 */
	public void setCartesian(int dim, Number v) {
		if (v instanceof Rational) {
			Rational coord = (Rational) v;
			if(dim==0)
				this.x.setRational(coord.numerator(), coord.denominator());
			else
				this.y.setRational(coord.numerator(), coord.denominator());
			return;
		}
		else if (v instanceof Integer) {
			Integer coord = (Integer) v;
			if(dim==0)
				this.x=new Rational(coord.intValue(), 1);
			else
				this.y=new Rational(coord.intValue(), 1);
			return;
		}
		throw new RuntimeException ("Wrong type number: method setY() is processing an object of type " + x.getClass());  	
	}

	/** 
	 * Set the coordinates of the current point to be (0, 0)
	 */
	public void setOrigin() {
		this.x=new Rational(0, 1);
		this.y=new Rational(0, 1);
	}

	/**
	 * Return the integer grid vector <tt>(p-q)</tt>, where <tt>q</tt> is the current point <br>
	 * <br>
	 * Warning: <tt>q</tt> is assumed to be an integer <tt>GridPoint_2</tt>
	 * 
	 * @return  a new <tt>GridVector_2</tt> vector corresponding to <tt>(p-q)</tt>
	 */
	public Vector_ minus(Point_ q){
		throw new Error("Not implemented yet");
	}

	public RationalPoint_2 sum(Vector_ v) {
		throw new Error("Not implemented yet");
	}

	/**
	 * Compare two points (lexicographic order on coordinates) <br>
	 * 
	 * @param  q the point to compare to the current point (this)
	 * @return  -1 if the current point is smaller (with respect to lexicographic order) to the point <tt>q</tt>;
	 * it returns 0 is the two points have equal coordinates. It returns 1 otherwise.
	 */
	public int compareTo(Point_ q) {
		if (q instanceof RationalPoint_2) {	  
			RationalPoint_2 p = (RationalPoint_2) q;

			int compareX=this.compareCartesian(p, 0);
			if(compareX!=0)
				return compareX;
			else
				return this.compareCartesian(p, 1);
		}
		else if(q instanceof GridPoint_2) {
			RationalPoint_2 p = new RationalPoint_2((GridPoint_2)q);

			int compareX=this.compareCartesian(p, 0);
			if(compareX!=0)
				return compareX;
			else
				return this.compareCartesian(p, 1);
			
		}
		throw new RuntimeException ("Method compareTo: comparing RationalPoint_2 with object of type " + q.getClass());  	
	}

	/**
	 * Compare the i-th coordinate of two points
	 * 
	 * @param o the point to compare
	 * @param i the i-th coordinate to compare
	 * @return -1 if the i-th coordinate of the current point <tt>this</tt> is smaller then i-th coordinate of point o, 
	 * return 0 if both points have the same i-th coordinate, and return 1 otherwise
	 */
	public int compareCartesian(Point_ o, int i) {
		if(i<0 || i>2)
			throw new Error("Error: wrong dimension "+i);
		if (o instanceof RationalPoint_2) {	  
			RationalPoint_2 p = (RationalPoint_2) o;
			if(this.getCartesian(i).doubleValue()<p.getCartesian(i).doubleValue()) // LCA: check, possible problem to to type conversion
				return -1;
			if(this.getCartesian(i).doubleValue()>p.getCartesian(i).doubleValue())
				return 1;
			return 0;
		}
		throw new RuntimeException ("Method compareCartesian: comparing RationalPoint_2 with object of type " + o.getClass());  	
	}

}
