package Jcg.geometry;

/**
 * A class for representing a 2D point with real coordinates.
 * 
 * @author Luca Castelli Aleardi (Ecole Polytechnique, 2010-2022)
 *
 */
public class Point_2 implements Point_ {
	/** an integer useful for indexing this point */
	public int index=-1; // by default, the index is undefined when set to -1

	/** Geometric coordinates of the point */
	public Double x,y;

	public Point_2() {}

	public Point_2(Number x,Number y) { 
		this.x=x.doubleValue(); 
		this.y=y.doubleValue();
	}

	/** 
	 * Initialize a 2D point having real coordinates <tt>(x, y)</tt>, and with a given <tt>index</tt> 
	 **/
	public Point_2(Number x,Number y, int index) { 
		this.x=x.doubleValue(); 
		this.y=y.doubleValue();
		this.index=index;
	}

	/** 
	 * Initialize a 2D point having real coordinates <tt>(0.0, 0.0)</tt>. 
	 **/
	public Point_2(Point_ p) { 
		this.x=p.getCartesian(0).doubleValue(); 
		this.y=p.getCartesian(1).doubleValue(); 
	}

	/**
	 * Set the current point as the barycenter of an array of points
	 */	  
	public void barycenter(Point_[] points) {
		double x_=0., y_=0.;
		for(int i=0;i<points.length;i++) {
			x_=x_+points[i].getCartesian(0).doubleValue();
			y_=y_+points[i].getCartesian(1).doubleValue();
		}
		this.x = x_/points.length;
		this.y = y_/points.length;
	}

	public void linearCombination(Point_[] points, Number[] coefficients) {
		double x_=0., y_=0.;
		for(int i=0;i<points.length;i++) {
			x_=x_+(points[i].getCartesian(0).doubleValue()*coefficients[i].doubleValue());
			y_=y_+(points[i].getCartesian(1).doubleValue()*coefficients[i].doubleValue());
		}
		this.x = x_;
		this.y = y_;
	}

	/**
	 * Return a new point defined by the linear combination of the input <tt>points</tt>,
	 * with respect to the given input <tt>coefficients</tt>
	 */
	public static Point_2 linearCombination(Point_2 [] points, Number[] coefficients) {
		Double x_=0., y_=0.;
		for(int i=0;i<points.length;i++) {
			x_=x_+(points[i].getX().doubleValue()*coefficients[i].doubleValue());
			y_=y_+(points[i].getY().doubleValue()*coefficients[i].doubleValue());
		}
		return new Point_2(x_,y_);
	}

	public static Point_2 midPoint (Point_2 p, Point_2 q) {
		Point_2 [] seg = new Point_2 [2];
		seg[0] = p;
		seg[1] = q;
		Number [] coef = new Number [2];
		coef[0] = coef[1] = 0.5;
		return linearCombination (seg, coef);
	}

	/** Return the index */
	public int getIndex() {return index; }
	public Double getX() {return x; }
	public Double getY() {return y; }

	/** Set the index */
	public void setIndex(int index) {this.index=index; }
	public void setX(Double x) {this.x=x.doubleValue(); }
	public void setY(Double y) {this.y=y.doubleValue(); }

	public void translateOf(Vector_ v) {
		this.x=x+v.getCartesian(0).doubleValue();
		this.y=y+v.getCartesian(1).doubleValue();
	}

	public boolean equals(Object o) {
		if (o instanceof Point_) {
			Point_ p = (Point_) o;
			return this.x.equals(p.getCartesian(0)) && this.y.equals(p.getCartesian(1)); 
		}
		throw new RuntimeException ("Method equals: comparing Point_2 with object of type " + o.getClass());  	
	}

	public int hashCode () {
		return (int)(this.x + this.y);
	}

	public Number distanceFrom(Point_2 p) {
		double dX=p.getX().doubleValue()-x;
		double dY=p.getY().doubleValue()-y;
		return Math.sqrt(dX*dX+dY*dY);
	}

	public Number squareDistance(Point_ p) {
		double dX=p.getCartesian(0).doubleValue()-x;
		double dY=p.getCartesian(1).doubleValue()-y;
		return dX*dX+dY*dY;
	}

	public String toString() {return "("+x+","+y+")"; }
	public int dimension() { return 2;}

	public Number getCartesian(int i) {
		if(i==0) return x;
		return y;
	} 
	public void setCartesian(int i, Number x) {
		if(i==0) this.x=x.doubleValue();
		else this.y=x.doubleValue();
	}

	public void setOrigin() {
		this.x=0.;
		this.y=0.;
	}

	public Vector_2 minus(Point_ b){
		return new Vector_2(b.getCartesian(0).doubleValue()-x, 
				b.getCartesian(1).doubleValue()-y);
	}

	public Point_2 sum(Vector_ v) {
		return new Point_2(this.x+v.getCartesian(0).doubleValue(),
				this.y+v.getCartesian(1).doubleValue());  	
	}

	/**
	 * Compare two points (lexicographic order on coordinates)
	 * @param o the point to compare
	 */
	public int compareTo(Point_ o) {
		if (o instanceof Point_2) {	  
			Point_2 p = (Point_2) o;
			if(this.x<p.getX().doubleValue())
				return -1;
			if(this.x>p.getX().doubleValue())
				return 1;
			if(this.y<p.getY().doubleValue())
				return -1;
			if(this.y>p.getY().doubleValue())
				return 1;
			return 0;
		}
		throw new RuntimeException ("Method compareTo: comparing Point_2 with object of type " + o.getClass());  	
	}

	/**
	 * Compare the i-th coordinate of two points
	 * @param o the point to compare
	 * @param i the i-th coordinate to compare
	 * @return the result is -1 if first point is smaller then point o, return 1 is first point is greater then o, and 0 otherwise
	 */
	public int compareCartesian(Point_ o, int i) {
		if(i<0 || i>2)
			throw new Error("Error: wrong dimension "+i);
		if (o instanceof Point_2) {	  
			Point_2 p = (Point_2) o;
			if(this.getCartesian(i).doubleValue()<p.getCartesian(i).doubleValue())
				return -1;
			if(this.getCartesian(i).doubleValue()>p.getCartesian(i).doubleValue())
				return 1;
			return 0;
		}
		throw new RuntimeException ("Method compareCartesian: comparing Point_2 with object of type " + o.getClass());  	
	}

}




