package Jcg.geometry;

/**
 * A class for representing a 2D point on an integer grid.
 * 
 * @author Luca Castelli Aleardi (Ecole Polytechnique, december 2022)
 *
 */
public class GridPoint_2 implements Point_ {
	/** an integer useful for indexing this point: for instance, 'index' can be the number of a vertex */
	public int index;
	
	/** integer coordinates of the point on the grid */
	public int x,y;

	/** Initialize an empty point */
	public GridPoint_2() {}

	/** 
	 * Initialize an integer point having coordinates (x, y) 
	 **/
	public GridPoint_2(int x,int y) { 
		this.x=x; 
		this.y=y;
	}

	/** 
	 * Initialize an integer point having coordinates (x, y), and with a given <tt>index</tt> 
	 **/
	public GridPoint_2(int x,int y, int index) { 
		this.x=x; 
		this.y=y;
		this.index=index;
	}

	/** 
	 * Make a copy of a point <tt>q</tt> 
	 **/
	public GridPoint_2(GridPoint_2 p) { 
		this.x=p.x; 
		this.y=p.y; 
	}

	/** Return the index */
	public int getIndex() {return index; }
	/** Return the x-coordinate */
	public int getX() {return x; }
	/** Return the y-coordinate */
	public int getY() {return y; }

	/** Set the index */
	public void setIndex(int index) {this.index=index; }
	/** Set the x-coordinate */
	public void setX(Number x) {this.x=x.intValue(); }
	/** Set the y-coordinate */
	public void setY(Number y) {this.y=y.intValue(); }

	/** 
	 * Translate the current point of the vector <tt>o</tt> 
	 **/
	public void translateOf(Vector_ o) {
		if (o instanceof GridVector_2) {
			GridVector_2 v = (GridVector_2) o;
			this.x=this.x+v.x;
			this.y=this.y+v.y;
			return;
		}
		throw new RuntimeException ("Error for method translateOf: translating a GridPoint_2 with object of type " + o.getClass());  	
	}

	/** 
	 * Check whether the current point is equal to point 'o'. <br>
	 * <br>
	 * Warning: point 'o' must be of type <tt>GridPoint_2</tt>
	 **/
	public boolean equals(Object o) {
		if (o instanceof GridPoint_2) {
			GridPoint_2 p = (GridPoint_2) o;
			return this.x==p.x && this.y==p.y; 
		}
		if (o instanceof RationalPoint_2) {
			RationalPoint_2 p = (RationalPoint_2) o;
			RationalPoint_2 current=new RationalPoint_2(this);
			return current.equals(p);
		}
		throw new RuntimeException ("Method equals: comparing GridPoint_2 with object of type " + o.getClass());  	
	}

	public int hashCode () {
		int max=10000;
		int xMod=x%max;
		return (int)(xMod*xMod + this.y);
	}

	public double distanceFrom(GridPoint_2 p) {
		return Math.hypot(this.x-p.x, this.y-p.y);
	}

	/**
	 * Return the square of the Euclidean distance between the current point 'this' 
	 * and a point <tt>p</tt>
	 */
	public Integer squareDistance(Point_ p) {
		int dX=p.getCartesian(0).intValue()-x;
		int dY=p.getCartesian(1).intValue()-y;
		return dX*dX+dY*dY;
	}

	/**
	 * Set the current point as the barycenter of an array of points
	 */	  
	public void barycenter(Point_[] points) {
		throw new Error("This method is not defined for the type GridPoint_2");
		/*double x_=0., y_=0.;
		for(int i=0;i<points.length;i++) {
			x_=x_+points[i].getCartesian(0).doubleValue();
			y_=y_+points[i].getCartesian(1).doubleValue();
		}
		this.x = x_/points.length;
		this.y = y_/points.length;*/
	}
	 
	/**
	 * Set the current point to be the linear combination of the input <tt>points</tt>,
	 * with respect to the given input <tt>coefficients</tt>
	 */
	public void linearCombination(Point_[] points, Number[] coefficients) {
		int x_=0, y_=0;
		for(int i=0;i<points.length;i++) {
			x_=x_+(points[i].getCartesian(0).intValue()*coefficients[i].intValue());
			y_=y_+(points[i].getCartesian(1).intValue()*coefficients[i].intValue());
		}
		this.x = x_;
		this.y = y_;
	}

	/**
	 * Return a new point defined by the linear combination of the input <tt>points</tt>,
	 * with respect to the given input <tt>coefficients</tt>
	 */
	public static GridPoint_2 linearCombination(GridPoint_2 [] points, Number[] coefficients) {
		Integer x_=0, y_=0;
		for(int i=0;i<points.length;i++) {
			x_=x_+(points[i].getX()*coefficients[i].intValue());
			y_=y_+(points[i].getY()*coefficients[i].intValue());
		}
		return new GridPoint_2(x_,y_);
	}

	public String toString() {return "("+x+","+y+")"; }
	public int dimension() { return 2;}

	/** 
	 * Return the d-th coordinate of the point <br>
	 * 
	 * @param dim  the d-th coordinate of the point: <br>
	 * dim=0 for x-coordinate, dim=1 for y-coordinate
	 */
	public Integer getCartesian(int dim) {
		if(dim==0) return x;
		return y;
	}

	/** 
	 * Set the d-th coordinate of the point 
	 * 
	 * @param dim  the d-th coordinate of the point: <br>
	 * dim=0 for x-coordinate, dim=1 for y-coordinate
	 */
	public void setCartesian(int dim, Number x) {
		if(dim==0) this.x=x.intValue();
		else this.y=x.intValue();
	}

	/** 
	 * Set the coordinates of the current point to be (0, 0)
	 */
	public void setOrigin() {
		this.x=0;
		this.y=0;
	}

	/**
	 * Return the integer grid vector <tt>(p-q)</tt>, where <tt>q</tt> is the current point <br>
	 * <br>
	 * Warning: <tt>q</tt> is assumed to be an integer <tt>GridPoint_2</tt>
	 * 
	 * @return  a new <tt>GridVector_2</tt> vector corresponding to <tt>(p-q)</tt>
	 */
	public GridVector_2 minus(Point_ q){
		return new GridVector_2(q.getCartesian(0).intValue()-this.x, q.getCartesian(1).intValue()-this.y);
	}

	/**
	 * @return true if CAB is a positive angle
	 */
	public static boolean ccw(GridPoint_2 A, GridPoint_2 B, GridPoint_2 C) {
		return C.minus(A).crossProduct(B.minus(A)) > 0;
	}

	/**
	 * @return true if segments AB and CD intersect
	 */
	public static boolean crossing(GridPoint_2 A, GridPoint_2 B, GridPoint_2 C, GridPoint_2 D) {
		if(A.equals(B) || A.equals(C) || A.equals(D) || B.equals(C) || B.equals(D) || C.equals(D) || A.minus(B).colinear(C.minus(D)) || A.minus(C).colinear(C.minus(D)) || C.minus(B).colinear(C.minus(D))) return false;
		return ccw(A,C,D) != ccw(B,C,D) && ccw(A,B,C) != ccw(A,B,D);
	}
	
	/**
	 * @return true if segment AB intersects with rectangle defined by C and D
	 */
	public static boolean crossingRect(GridPoint_2 A, GridPoint_2 B, GridPoint_2 C, GridPoint_2 D) {
		return crossing(A, B, C, new GridPoint_2(C.x, D.y)) || crossing(A, B, new GridPoint_2(C.x, D.y), D) || crossing(A, B, D, new GridPoint_2(D.x, C.y)) || crossing(A, B, C, new GridPoint_2(D.x, C.y));
	}

	public GridPoint_2 sum(GridVector_2 v) {
		return new GridPoint_2(this.x+v.x,this.y+v.y);  	
	}

	/**
	 * Compare two points (lexicographic order on coordinates) <br>
	 * 
	 * @param  q the point to compare to the current point (this)
	 * @return  -1 if the current point is smaller (with respect to lexicographic order) to the point <tt>q</tt>;
	 * it returns 0 is the two points have equal coordinates. It returns 1 otherwise.
	 */
	public int compareTo(Point_ q) {
		if (q instanceof GridPoint_2) {	  
			GridPoint_2 p = (GridPoint_2) q;
			if(this.x<p.x)
				return -1;
			if(this.x>p.x)
				return 1;
			if(this.y<p.y)
				return -1;
			if(this.y>p.y)
				return 1;
			return 0;
		}
		else if (q instanceof RationalPoint_2) {	  
			RationalPoint_2 p = (RationalPoint_2) q;
			RationalPoint_2 current=new RationalPoint_2(this);
			return current.compareTo(p);
		}
		throw new RuntimeException ("Method compareTo: comparing GridPoint_2 with object of type " + q.getClass());  	
	}

	/**
	 * Compare the i-th coordinate of two points
	 * 
	 * @param o the point to compare
	 * @param i the i-th coordinate to compare
	 * @return -1 if the i-th coordinate of the current point <tt>this</tt> is smaller then i-th coordinate of point o, 
	 * return 0 if both points have the same i-th coordinate, and return 1 otherwise
	 */
  	public int compareCartesian(Point_ o, int i) {
  		if(i<0 || i>2)
  			throw new Error("Error: wrong dimension "+i);
  		if (o instanceof GridPoint_2) {	  
		  GridPoint_2 p = (GridPoint_2) o;
		  if(this.getCartesian(i)<p.getCartesian(i))
			  return -1;
		  if(this.getCartesian(i)>p.getCartesian(i))
			  return 1;
		  return 0;
	  }
	  throw new RuntimeException ("Method compareCartesian: comparing GridPoint_2 with object of type " + o.getClass());  	
  	}

}




